from django.db.models import Prefetch, F
from rest_framework import serializers

from intranet.femida.src.api.candidates.serializers import ConsiderationLiteSerializer
from intranet.femida.src.api.core.serializers import (
    AwareSerializerMixin,
    FemidaSerializer,
    WorkflowActionsField,
)
from intranet.femida.src.api.users.serializers import UserSerializer
from intranet.femida.src.candidates.choices import CHALLENGE_STATUSES
from intranet.femida.src.candidates.helpers import get_main_email
from intranet.femida.src.candidates.models import Challenge, Candidate, Consideration
from intranet.femida.src.certifications.models import Certification
from intranet.femida.src.certifications.workflow import CertificationWorkflow
from intranet.femida.src.interviews.choices import INTERVIEW_STATES
from intranet.femida.src.interviews.models import Interview


class CandidateCertificationCreateFormSerializer(serializers.Serializer):

    receiver = serializers.SerializerMethodField()

    def get_receiver(self, obj: Candidate):
        return get_main_email(obj)


class CandidateCertificationDetailSerializer(FemidaSerializer):

    consideration = ConsiderationLiteSerializer(
        context={
            'fields': ('id', 'status', 'extended_status'),
        },
    )
    created_by = UserSerializer()

    class Meta:
        model = Certification
        fields = (
            'id',
            'consideration',
            'created_by',
        )


class InterviewForCertificationSerializer(FemidaSerializer):

    vacancy_id = serializers.ReadOnlyField()

    class Meta:
        model = Interview
        fields = (
            'id',
            'type',
            'aa_type',
            'grade',
            'resolution',
            'state',
            'created',
            'finished',
            'is_code',
            'vacancy_id',
        )


class ChallengeForCertificationSerializer(FemidaSerializer):

    class Meta:
        model = Challenge
        fields = (
            'id',
            'type',
            'resolution',
            'created',
            'finished',
            'status',
        )


class ConsiderationForCertificationSerializer(FemidaSerializer):

    class Meta:
        model = Consideration
        fields = (
            'started',
            'finished',
        )


class CandidateForCertificationSerializer(FemidaSerializer):

    class Meta:
        model = Candidate
        fields = (
            'first_name',
            'last_name',
        )


class CertificationSerializer(AwareSerializerMixin, FemidaSerializer):

    actions = WorkflowActionsField(workflow_class=CertificationWorkflow)
    consideration = ConsiderationForCertificationSerializer()
    candidate = CandidateForCertificationSerializer(source='consideration.candidate')
    interviews = InterviewForCertificationSerializer(
        source='consideration.finished_interviews',
        many=True,
    )
    challenges = ChallengeForCertificationSerializer(
        source='consideration.finished_challenges',
        many=True,
    )
    public_uuid = serializers.SerializerMethodField()

    def get_public_uuid(self, obj: Certification):
        if obj.is_published:
            return str(obj.public_uuid)
        return None

    class Meta:
        model = Certification
        fields = (
            'actions',
            'is_published',
            'public_uuid',
            'consideration',
            'candidate',
            'interviews',
            'challenges',
        )
        select_related_map = {
            'consideration': ('consideration',),
            'candidate': ('consideration__candidate',),
        }
        prefetch_related_map = {
            'interviews': (
                Prefetch(
                    lookup='consideration__interviews',
                    queryset=(
                        Interview.unsafe
                        .filter(state=INTERVIEW_STATES.finished)
                        .annotate(vacancy_id=F('application__vacancy_id'))
                        .order_by('finished')
                    ),
                    to_attr='finished_interviews',
                ),
            ),
            'challenges': (
                Prefetch(
                    lookup='consideration__challenges',
                    queryset=(
                        Challenge.objects
                        .filter(status=CHALLENGE_STATUSES.finished)
                        .order_by('finished')
                    ),
                    to_attr='finished_challenges',
                ),
            ),
        }


class CertificationPublicSerializer(CertificationSerializer):

    def to_representation(self, instance: Certification):
        if not instance.is_published:
            return {'is_published': False}
        return super().to_representation(instance)

    class Meta(CertificationSerializer.Meta):
        fields = (
            'is_published',
            'consideration',
            'candidate',
            'interviews',
            'challenges',
        )
