from rest_framework import serializers

from intranet.femida.src.api.core.fields import SafeWikiFormattedField
from intranet.femida.src.api.core.serializers import (
    FemidaSerializer,
    WorkflowActionsField,
    AwareSerializerMixin,
)
from intranet.femida.src.api.problems.serializers import ProblemSerializer
from intranet.femida.src.api.users.serializers import UserSerializer
from intranet.femida.src.problems.complaints.workflow import ComplaintWorkflow
from intranet.femida.src.problems.models import Complaint


class ComplaintLiteSerializer(FemidaSerializer):

    actions = WorkflowActionsField(ComplaintWorkflow)

    class Meta:
        model = Complaint
        fields = (
            'id',
            'is_active',
            'resolution',
            'actions',
        )


class ComplaintSerializer(AwareSerializerMixin, FemidaSerializer):

    problem = ProblemSerializer()
    original_problem = ProblemSerializer()
    created_by = UserSerializer()
    kind_verbose = serializers.SerializerMethodField()
    formatted_comment = SafeWikiFormattedField()
    actions = WorkflowActionsField(ComplaintWorkflow)
    comments_count = serializers.SerializerMethodField()

    def get_kind_verbose(self, obj):
        return obj.get_kind_display()

    def get_comments_count(self, obj):
        # Аттрибута comments_count нет только в момент
        # создания жалобы, поэтому можно считать, что там 0
        return getattr(obj, 'comments_count', 0)

    class Meta:
        model = Complaint
        fields = (
            'id',
            'problem',
            'original_problem',
            'is_active',
            'created_by',
            'comment',
            'formatted_comment',
            'kind',
            'kind_verbose',
            'resolution',
            'created',
            'actions',
            'comments_count',
        )
        select_related_map = {
            'created_by': ('created_by',),
        }
        prefetch_related_map = {
            'problem': (
                'problem__created_by',
                'problem__categories',
            ),
            'original_problem': (
                'original_problem__created_by',
                'original_problem__categories',
            ),
        }


class ComplaintListFilterFormSerializer(serializers.Serializer):

    moderator = serializers.ReadOnlyField(source='moderator.username')
    category = serializers.ReadOnlyField(source='category.id')
