import json

from constance import config
from django.db.models import OuterRef, F, Subquery
from django.utils import timezone

from intranet.femida.src.candidates.choices import SLA_STATUSES
from intranet.femida.src.candidates.helpers import (
    candidate_is_current_employee_subquery,
    get_extended_status_changed_at_subquery,
)
from intranet.femida.src.candidates.models import Candidate, ConsiderationIssue
from intranet.femida.src.core.models import City
from intranet.femida.src.core.db import RowToDict, RowsToList as Rows
from intranet.femida.src.interviews.models import Interview
from intranet.femida.src.professions.models import Profession
from intranet.femida.src.utils.translation import get_name_field, get_localized_name_field


Row = lambda x: RowToDict(x[:1])


class ConsiderationDashboardListSerializer:
    """
    Сериализатор рассмотрений для дашборда, мимикрирующий под списковый DRF-сериализатор
    """
    def __init__(self, instance, *args, **kwargs):
        self.conf_extended_status_sla = json.loads(config.EXTENDED_STATUS_SLA)
        self.user = kwargs['context']['user']
        self.data = [self._clean_consideration(i) for i in instance]

    def _clean_consideration(self, consideration):
        consideration['candidate'] = candidate = consideration.pop('candidate_')
        consideration['interviews'] = consideration.pop('interviews_') or []
        consideration['issues'] = consideration.pop('issues_') or []
        candidate['target_cities'] = candidate.pop('target_cities_') or []
        candidate['professions'] = candidate.pop('professions_') or []
        for profession in candidate['professions']:
            profession['name'] = profession.pop('name_')

        consideration['sla_status'] = None
        now = timezone.now()
        changed_at = consideration['extended_status_changed_at']
        status_days = (now - changed_at).days
        sla_conf = self.conf_extended_status_sla.get(consideration['extended_status'], {})
        danger_days = sla_conf.get(SLA_STATUSES.danger, None)
        warning_days = sla_conf.get(SLA_STATUSES.warning, None)

        if warning_days is not None and status_days >= warning_days:
            consideration['sla_status'] = SLA_STATUSES.warning
        if danger_days is not None and status_days >= danger_days:
            consideration['sla_status'] = SLA_STATUSES.danger

        return consideration

    @classmethod
    def setup_eager_loading(cls, queryset, *args, **kwargs):
        return cls._get_consideration_queryset(queryset)

    @classmethod
    def _get_consideration_queryset(cls, queryset=None):
        return queryset.values(
            'id',
            'state',
            'extended_status',
            'resolution',
            'finished',
            extended_status_changed_at=Subquery(get_extended_status_changed_at_subquery()[:1]),
            candidate_=Row(cls._get_candidate_subquery()),
            interviews_=Rows(cls._get_interviews_subquery()),
            issues_=Rows(cls._get_consideration_issues_subquery()),
        )

    @classmethod
    def _get_candidate_subquery(cls):
        return (
            Candidate.unsafe
            .filter(id=OuterRef('candidate_id'))
            .values(
                'id',
                'first_name',
                'last_name',
                'login',
                is_current_employee=candidate_is_current_employee_subquery,
                target_cities_=Rows(cls._get_target_cities_subquery()),
                professions_=Rows(cls._get_professions_subquery()),
            )
        )

    @classmethod
    def _get_interviews_subquery(cls):
        return (
            Interview.unsafe
            .filter(consideration_id=OuterRef('id'))
            .alive()
            .values(
                'id',
                'type',
                'grade',
                'yandex_grade',
                'state',
                'resolution',
            )
        )

    @classmethod
    def _get_target_cities_subquery(cls):
        return (
            City.objects
            .filter(candidates=OuterRef('id'))
            .values('id', name=F(get_name_field()))
        )

    @classmethod
    def _get_professions_subquery(cls):
        return (
            Profession.objects
            .filter(candidate_professions__candidate=OuterRef('id'))
            .values('id', name_=F(get_localized_name_field()))
        )

    @classmethod
    def _get_consideration_issues_subquery(cls):
        return (
            ConsiderationIssue.objects
            .filter(consideration_id=OuterRef('id'))
            .exclude(is_resolved=True)
            .values(
                'id',
                'type',
                'level',
                'params',
            )
        )
