from django.conf import settings
from django.db.models import Prefetch, Count
from rest_framework import serializers

from intranet.femida.src.api.assignments.serializers import AssignmentLiteSerializer
from intranet.femida.src.api.challenges.serializers import ChallengeBaseSerializer
from intranet.femida.src.api.core.fields import SafeWikiFormattedField
from intranet.femida.src.api.core.serializers import (
    FemidaSerializer,
    IdNameSerializer,
    WorkflowActionsField,
    AwareSerializerMixin,
)
from intranet.femida.src.api.users.serializers import UserSerializer
from intranet.femida.src.candidates.models import Consideration, Challenge, Candidate
from intranet.femida.src.candidates.choices import CHALLENGE_STATUSES
from intranet.femida.src.interviews.helpers import (
    InterviewVisibilityHelper,
    can_user_see_review_issue,
)
from intranet.femida.src.contest.converter import convert_contest_info_to_json_dict
from intranet.femida.src.interviews.choices import INTERVIEW_STATES, INTERVIEW_ROUND_STATUSES
from intranet.femida.src.interviews.models import Interview, InterviewRound
from intranet.femida.src.interviews.workflow import InterviewWorkflow, InterviewRoundWorkflow


class InterviewForConsiderationSerializer(FemidaSerializer):

    actions = WorkflowActionsField(InterviewWorkflow)
    vacancy = IdNameSerializer(source='application.vacancy', default=None)
    created_by = UserSerializer()
    interviewer = UserSerializer()
    grade_verbose = serializers.ReadOnlyField(source='get_grade_display')
    assignments = AssignmentLiteSerializer(many=True)
    formatted_comment = SafeWikiFormattedField()
    startrek_review_key = serializers.SerializerMethodField()

    comments_count = serializers.IntegerField()

    def __init__(self, *args, **kwargs):
        """
        TODO: Эту логику неплохо бы потом запихать в какой-нибудь базовый
        сериализатор
        """
        fields = kwargs.pop('fields', None)
        super().__init__(
            *args, **kwargs)
        if fields:
            allowed = set(fields)
            existing = set(self.fields.keys())
            for field_name in existing - allowed:
                self.fields.pop(field_name)

    def get_startrek_review_key(self, obj):
        user = self.root.context.get('user')
        if user and can_user_see_review_issue(user, obj):
            return obj.startrek_review_key

    class Meta:
        model = Interview
        fields = (
            'id',
            'type',
            'actions',
            'assignments',
            'comment',
            'formatted_comment',
            'created',
            'created_by',
            'grade',
            'grade_verbose',
            'yandex_grade',
            'event_id',
            'event_start_time',
            'finished',
            'interviewer',
            'section',
            'state',
            'resolution',
            'vacancy',
            'aa_type',
            'is_code',
            'is_pro_level_scale',
            'comments_count',
            'startrek_review_key',
        )


class ChallengeForConsiderationSerializer(ChallengeBaseSerializer):

    comments_count = serializers.IntegerField()
    contest_url = serializers.SerializerMethodField()
    contest_info = serializers.SerializerMethodField()

    def get_contest_info(self, obj):
        return convert_contest_info_to_json_dict(obj.contest, obj.contest_participant)

    def get_contest_url(self, obj):
        participant_id = obj.participant_id
        if not participant_id:
            return None
        return f'{settings.CONTEST_INTERVIEW_URL}participants/{participant_id}'

    class Meta(ChallengeBaseSerializer.Meta):
        fields = ChallengeBaseSerializer.Meta.fields + (
            'comments_count',
            'contest_url',
            'contest_info'
        )


class CandidateForConsiderationSerializer(FemidaSerializer):
    """
    Используется для кандидатов внутри рассмотрений
    """
    class Meta:
        model = Candidate
        fields = (
            'id',
            'full_name',
            'login',
            'startrek_key',
        )


class InterviewRoundSerializer(FemidaSerializer):

    actions = WorkflowActionsField(InterviewRoundWorkflow)
    created_by = UserSerializer()
    interviews = InterviewForConsiderationSerializer(many=True, fields=(
        'id',
        'section',
        'type',
        'aa_type',
        'is_code',
        'vacancy',
    ))

    class Meta:
        model = InterviewRound
        fields = (
            'id',
            'actions',
            'type',
            'created_by',
            'created',
            'interviews',
        )


class ConsiderationSerializer(AwareSerializerMixin, FemidaSerializer):

    created_by = UserSerializer()
    challenges = ChallengeForConsiderationSerializer(many=True)
    interviews = serializers.SerializerMethodField()
    interview_rounds = InterviewRoundSerializer(many=True)

    def get_interviews(self, obj):
        """
        Не получилось по простому переопределить сам сериализатор собеседования,
        потому что в ListSerializer используется один инстанс дочернего
        сериализатора, а fields у сериализатора кэшируется на уровне инстанса
        """
        interview_serializer_lite = InterviewForConsiderationSerializer(
            fields=('id', 'type', 'actions', 'created', 'created_by', 'finished',
                    'event_id', 'event_start_time', 'aa_type', 'is_code',
                    'section', 'state', 'vacancy', 'interviewer'),
            context=self.context,
        )
        interview_serializer_full = InterviewForConsiderationSerializer(
            context=self.context,
        )

        # Это обязательно, иначе не будут оборачиваться юзеры отвтом Staff API
        interview_serializer_lite.bind('', self)
        interview_serializer_full.bind('', self)

        request_user = self.context.get('user')
        visibility_helper = InterviewVisibilityHelper(request_user, obj)

        result = []
        for interview in obj.interviews.all():
            if not visibility_helper.is_visible(interview):
                serialized_interview = interview_serializer_lite.to_representation(interview)
                serialized_interview['is_full_data'] = False
            else:
                serialized_interview = interview_serializer_full.to_representation(interview)
                serialized_interview['is_full_data'] = True

            result.append(serialized_interview)

        state_order = {s: i for i, (s, _) in enumerate(INTERVIEW_STATES)}
        return sorted(
            result,
            key=lambda x: (
                -state_order.get(x['state']),
                x['finished'] or '',
                x['event_start_time'] or '',
            ),
            reverse=True,
        )

    class Meta:
        model = Consideration
        fields = (
            'id',
            'state',
            'status',
            'created_by',
            'created',
            'started',
            'finished',
            'estimation_scale_type',
            'challenges',
            'interviews',
            'interview_rounds',
        )
        select_related_map = {
            'candidate': ('candidate',),
            'created_by': ('created_by',),
        }
        # TODO: Есть потенциальная проблема с фильтрацией в prefetch_related,
        # если захочется использовать сериализатор в отрыве от BaseView.
        # Потому что кроме неоптимальных запросов мы получим еще и отличающуюся выборку,
        # так как interviews.all() будет отдавать неотфильтрованные данные.
        prefetch_related_map = {
            'challenges': (
                Prefetch(
                    lookup='challenges',
                    queryset=(
                        Challenge.objects
                        .exclude(status=CHALLENGE_STATUSES.cancelled)
                        .annotate(comments_count=Count('comments'))
                        .select_related(
                            'reviewed_by',
                            'submission__form',
                        )
                        .order_by('-created')
                    ),
                ),
                'challenges__submission__vacancies',
            ),
            'interviews': (
                Prefetch(
                    lookup='interviews',
                    queryset=(
                        Interview.unsafe.alive()
                        .annotate(comments_count=Count('comments'))
                        .select_related(
                            'application__vacancy',
                            'created_by',
                            'interviewer',
                        )
                        .prefetch_related(
                            'assignments__problem',
                        )
                    ),
                ),
            ),
            'interview_rounds': (
                Prefetch(
                    lookup='interview_rounds',
                    queryset=(
                        InterviewRound.objects
                        .filter(status__in=[
                            INTERVIEW_ROUND_STATUSES.new,
                            INTERVIEW_ROUND_STATUSES.planning,
                        ])
                        .select_related(
                            'created_by',
                        )
                        .prefetch_related(
                            'interviews__application__vacancy',
                        )
                        .order_by('-created')
                    ),
                ),
            ),
        }
