from django.core.paginator import (
    EmptyPage,
    Page,
    PageNotAnInteger,
    Paginator as DjangoPaginator,
)
from django.utils.translation import ugettext_lazy as _

from rest_framework import pagination
from rest_framework.pagination import _positive_int


class PageNumberPagination(pagination.PageNumberPagination):

    page_size_query_param = 'page_size'
    page_size = 1000


class ISearchPaginator(DjangoPaginator):

    def __init__(self, object_list, *args, **kwargs):
        """
        object_list - коллекция ISearchCollection
        """
        super().__init__(object_list, *args, **kwargs)

    def validate_number(self, number):
        """
        Чтобы не делать лишний запрос на count, убираем из оригинального validate_number
        проверку на то, что number > num_pages
        """
        try:
            number = int(number)
        except (TypeError, ValueError):
            raise PageNotAnInteger(_('That page number is not an integer'))
        if number < 1:
            raise EmptyPage(_('That page number is less than 1'))
        return number

    def page(self, number):
        number = self.validate_number(number)
        return Page(
            object_list=self.object_list.get_page(number),
            number=number,
            paginator=self,
        )


class ISearchPageNumberPagination(PageNumberPagination):
    """
    Фейковый пагинатор для поисковых, который ничего не пагинирует,
    но формирует стандартный пагинированный ответ
    """
    page_size = 20
    max_page_size = 20
    django_paginator_class = ISearchPaginator


class CursorPagination(pagination.CursorPagination):

    page_size = 50
    max_page_size = 10000
    page_size_query_param = 'page_size'
    ordering = '-id'

    def get_page_size(self, request):
        try:
            return _positive_int(
                request.query_params[self.page_size_query_param],
                strict=True,
                cutoff=self.max_page_size,
            )
        except (KeyError, ValueError):
            return self.page_size

    def get_ordering(self, request, queryset, view):
        """
        Получаем сортировку для пагинатора из исходного View
        """
        if not hasattr(view, 'get_ordering'):
            return super().get_ordering(request, queryset, view)

        ordering = view.get_ordering()
        assert ordering

        if isinstance(ordering, str):
            ordering = (ordering,)
        if isinstance(ordering, list):
            ordering = tuple(ordering)
        return ordering


class MagicLinksPagination(CursorPagination):

    page_size = 100
    max_page_size = 100


class JobsPagination(CursorPagination):

    page_size = 200


class ExternalPublicationPagination(CursorPagination):

    ordering = ('-priority', '-published_at', '-id')

    def get_ordering(self, request, queryset, view):
        """
        Для случая, когда осуществлялся полнотекстовый поиск, queryset аннотирован полем rank, по
        которому надо сортировать внутри сортировки по priority (прибитой гвоздями в админке)
        """
        ordering = super().get_ordering(request, queryset, view)
        if 'rank' in queryset.query.annotations:
            ordering = ordering[0], '-rank', *ordering[1:]

        return ordering


class CandidateFilterPagination(CursorPagination):

    page_size = 20
    max_page_size = 100
