from django.db import models
from django.db.models.query import prefetch_related_objects

from rest_framework import serializers

from intranet.femida.src.utils.translation import get_localized_field


class AwareSerializerMixin:
    """
    Добавляет в сериализатор поддержку select/prefetch_related и указания подмножества полей.

    Позволяет указывать related-поля в теле сериализатора, а не во view.
    Для этого в подклассе Meta необходимо определить аттрибуты select_related_map,
    prefetch_related_map.

    Позволяет при создании сериализатора определить подмножество полей для отображения.
    При удалении лишних полей из объекта удаляются и относящиеся к ним select/prefetch.
    """

    def __init__(self, *args, **kwargs):
        """
        Удаляем лишние поля из сериализатора
        """
        super().__init__(*args, **kwargs)

        # FIXME: плохо использовать context в конструкторе
        # `context` - cached_property, внутри происходит след.:
        # в property `context` идет обращение к `root`,
        # который в свою очередь обращается к `parent`,
        # а на момент __init__ `parent` еще не проставлен,
        # он проставляется позже через `bind(...)`.
        # Как следствие, мы кэшируем пустой dict.
        # В качестве временного решения, обращаемся к `_context`,
        # т.е. это будет работать только для верхнеуровневых сериализаторов.
        optional_fields = self._context.get('fields', [])
        if not optional_fields:
            return

        optional_fields = set(optional_fields)

        for field in list(self.fields):
            if field not in optional_fields:
                self.fields.pop(field)

    @classmethod
    def setup_eager_loading(cls, collection, fields=None):
        """
        Делаем всякие select_related и prefetch_related
        """
        fields = set(fields) if fields else set()
        all_fields = not fields

        is_queryset = isinstance(collection, models.QuerySet)

        meta = getattr(cls, 'Meta', None)
        select_related_map = getattr(meta, 'select_related_map', {})
        prefetch_related_map = getattr(meta, 'prefetch_related_map', {})

        if is_queryset:
            select_related_fields = []
            for field in select_related_map:
                if field in fields or all_fields:
                    select_related_fields.extend(select_related_map[field])

            if select_related_fields:
                collection = collection.select_related(*select_related_fields)

        prefetch_related_fields = []
        for field in prefetch_related_map:
            if field in fields or all_fields:
                prefetches = prefetch_related_map[field]
                # Note: в качестве префетча можно использовать callable, если нужен ленивый префетч
                prefetch_related_fields.extend(p() if callable(p) else p for p in prefetches)

        if prefetch_related_fields:
            if is_queryset:
                collection = collection.prefetch_related(*prefetch_related_fields)
            else:
                prefetch_related_objects(collection, *prefetch_related_fields)

        return collection


class FemidaSerializer(serializers.ModelSerializer):

    @property
    def errors(self):
        ret = super().errors
        if ret:
            ret['errors'] = dict(ret)
        return ret


class IdNameSerializer(serializers.Serializer):

    id = serializers.ReadOnlyField()
    name = serializers.SerializerMethodField()

    def __init__(self, *args, name_source='name', **kwargs):
        self.name_source = name_source
        super().__init__(*args, **kwargs)

    def get_name(self, obj):
        return getattr(obj, self.name_source)


class IdSlugNameSerializer(IdNameSerializer):

    slug = serializers.ReadOnlyField()


class WorkflowActionsField(serializers.Field):

    def __init__(self, workflow_class=None, **kwargs):
        self.workflow_class = workflow_class
        kwargs['source'] = '*'
        kwargs['read_only'] = True
        super().__init__(**kwargs)

    def to_representation(self, value):
        user = self.root.context.get('user')
        if user:
            return self.workflow_class(
                instance=value,
                user=user,
            ).get_actions_visibility()
        return {}


class LocalizedField(serializers.ReadOnlyField):

    def __init__(self, source=None):
        super().__init__(source='*')
        self._source = source

    def get_attribute(self, instance):
        source = self._source or self.field_name
        self.source = get_localized_field(source)
        self.source_attrs = self.source.split('.')
        return super().get_attribute(instance)
