from django.db.models import Q
from django.contrib.auth import get_user_model
from django.forms import BooleanField

from intranet.femida.src.api.duplication_cases import serializers
from intranet.femida.src.api.core.views import BaseView, WorkflowView
from intranet.femida.src.candidates.choices import DUPLICATION_CASE_STATUSES
from intranet.femida.src.candidates.models import DuplicationCase
from intranet.femida.src.candidates.workflow import DuplicationCaseWorkflow


User = get_user_model()


class DuplicationCaseQuerysetOptimizerMixin:

    def get_queryset(self):
        # TODO: Это все по-хорошему надо переписать. Запросы плохие получаются.
        return (
            self.model_class.objects
            .prefetch_related(
                'first_candidate__candidate_attachments__attachment',
                'first_candidate__contacts',
                'second_candidate__candidate_attachments__attachment',
                'second_candidate__contacts',
            )
            .select_related(
                'merged_to',
                'submission',
                'managed_by',
            )
            .order_by('-created')
        )


class DuplicationCaseListView(DuplicationCaseQuerysetOptimizerMixin, BaseView):

    model_class = DuplicationCase
    list_item_serializer_class = serializers.DuplicationCaseSerializer

    def get(self, request, *args, **kwargs):
        """
        Список случаев дубликации кандидатов
        """
        return self.list(request, *args, **kwargs)

    def filter_queryset(self, queryset):
        params = self.request.query_params
        is_open = BooleanField().to_python(params['is_open']) if 'is_open' in params else None
        candidate_id = params.get('candidate')

        is_open_status_map = {
            True: DUPLICATION_CASE_STATUSES.new,
            False: DUPLICATION_CASE_STATUSES.closed,
        }
        if is_open is not None:
            queryset = queryset.filter(status=is_open_status_map[is_open])

        if candidate_id is not None:
            queryset = queryset.filter(
                Q(first_candidate_id=candidate_id)
                | Q(second_candidate_id=candidate_id)
            )

        return queryset


class DuplicationCaseDetailView(DuplicationCaseQuerysetOptimizerMixin, BaseView):

    model_class = DuplicationCase
    detail_serializer_class = serializers.DuplicationCaseSerializer

    def get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)


class DuplicationCaseCancelView(WorkflowView):

    model_class = DuplicationCase
    workflow_class = DuplicationCaseWorkflow
    detail_serializer_class = serializers.DuplicationCaseSerializer
    actionlog_name = 'duplication_case_cancel'
    action_name = 'cancel'


class DuplicationCaseMarkUnclearView(WorkflowView):

    model_class = DuplicationCase
    workflow_class = DuplicationCaseWorkflow
    detail_serializer_class = serializers.DuplicationCaseSerializer
    actionlog_name = 'duplication_case_mark_unclear'
    action_name = 'mark_unclear'
