import mimetypes
import os

from base64 import b64decode

from django.http import Http404, HttpResponse, HttpResponseRedirect
from django.utils.decorators import method_decorator
from django_replicated.decorators import use_master
from rest_framework import status

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.api.core import errors
from intranet.femida.src.api.core.serializers import IdNameSerializer
from intranet.femida.src.api.core.views import BaseView, ResponseError
from intranet.femida.src.attachments.models import Attachment
from intranet.femida.src.hh.api import HeadhunterAPI
from intranet.femida.src.hh.choices import HH_RESUME_FILE_TYPES
from intranet.femida.src.hh.exceptions import HeadhunterError, HeadhunterOauthError
from intranet.femida.src.hh.models import HHOauthToken
from intranet.femida.src.hh.oauth import HeadhunterOAuth

from . import forms

if os.getenv('IS_ARCADIA'):
    from django_mds.client import APIError
else:
    from mds import APIError


class AuthorizeView(BaseView):

    @method_decorator(use_master)
    def get(self, request, *args, **kwargs):
        code = request.query_params.get('code')
        if not code:
            return HttpResponseRedirect('/')

        HeadhunterOAuth.generate_token(self.request.user, code)

        state = request.query_params.get('state')
        return_path = b64decode(state) if state else '/'
        return HttpResponseRedirect(return_path)


class TokenRefreshView(BaseView):

    @method_decorator(use_master)
    def get(self, request, *args, **kwargs):
        state = request.query_params.get('state')
        return_path = b64decode(state) if state else '/'
        try:
            token = HHOauthToken.objects.get(user=self.request.user)
            HeadhunterOAuth.refresh_token(token)
        except (HeadhunterOauthError, HHOauthToken.DoesNotExist):
            return_path = HeadhunterOAuth.get_oauth_url(return_path)
        return HttpResponseRedirect(return_path)


class ResumeFileView(BaseView):

    def get(self, request, *args, **kwargs):
        """
        Просмотр файла резюме с HH
        """
        file_type = kwargs['file_type']
        if file_type not in HH_RESUME_FILE_TYPES:
            raise Http404

        hh = HeadhunterAPI(self.request.user)
        try:
            resume_file = hh.download_resume_by_type(kwargs['resume_id'], file_type)
        except HeadhunterError as exc:
            raise ResponseError(
                data={'error': [{'code': exc.message}]},
                status=status.HTTP_400_BAD_REQUEST,
            )

        file_name = 'resume.{}'.format(file_type)
        content_type = mimetypes.guess_type(file_name)[0]
        response = HttpResponse(resume_file, content_type=content_type or 'text/plain')
        response['Content-Disposition'] = 'filename="{}"'.format(file_name)
        return response


class ResumeUploadView(BaseView):

    validator_class = forms.ResumeUploadForm
    detail_serializer_class = IdNameSerializer

    @action_logged('hh_resume_upload')
    def post(self, request, *args, **kwargs):
        """
        Заливка резюме HH
        """
        return self.create(request, *args, **kwargs)

    def perform_create(self, data):
        hh = HeadhunterAPI(self.request.user)
        try:
            hh_data = hh.get_resume(data['resume_id'])
            attached_file = hh.download_resume_by_type(data['resume_id'], data['type'])
            name_fields = ('first_name', 'middle_name', 'last_name')
            name = ' '.join(hh_data[f] for f in name_fields if hh_data.get(f))
            return Attachment.objects.create(
                name=f'{name}.{data["type"]}',
                uploader=self.request.user,
                attached_file=attached_file,
            )
        except HeadhunterError as exc:
            raise ResponseError(
                data={'error': [{'code': exc.message}]},
                status=status.HTTP_400_BAD_REQUEST,
            )
        except APIError:
            raise ResponseError(
                data=errors.format_non_field_message(errors.UPLOAD_TO_MDS_FAILED),
                status=status.HTTP_400_BAD_REQUEST,
            )
