import hashlib
import json
import sform

from constance import config
from django.conf import settings
from django.utils.formats import ISO_INPUT_FORMATS

from intranet.femida.src.api.core.forms import (
    ConditionallyRequiredFieldsMixin,
    CurrencySuggestField,
    DepartmentSuggestField,
    PositionSuggestField,
    PositiveDecimalField,
    PositiveIntegerField,
    ProfessionSuggestField,
    ServiceMultipleSuggestField,
    UserSuggestField,
    validate_recruiters,
)
from intranet.femida.src.api.core.validators import validate_phone
from intranet.femida.src.candidates.contacts import normalize_contact
from intranet.femida.src.offers import choices as offer_choices
from intranet.femida.src.professions.models import Profession
from intranet.femida.src.staff.models import Department, Organization, Office
from intranet.femida.src.utils.strings import fetch_comma_separated_integers
from intranet.femida.src.vacancies.choices import VACANCY_TYPES


class HireOrderCandidateForm(sform.SForm):

    first_name = sform.CharField(max_length=255, state=sform.REQUIRED)
    last_name = sform.CharField(max_length=255, state=sform.REQUIRED)
    email = sform.EmailField(max_length=255, state=sform.REQUIRED)
    phone = sform.CharField(
        max_length=255,
        validators=[validate_phone],
    )
    source = sform.ChoiceField(offer_choices.SOURCES, state=sform.REQUIRED)
    source_description = sform.CharField(max_length=255)

    def clean(self):
        cleaned_data = super().clean()
        cleaned_data['contacts'] = []
        for field in ('email', 'phone'):
            account_id = cleaned_data.pop(field, '')
            if account_id:
                cleaned_data['contacts'].append({
                    'type': field,
                    'is_main': True,
                    'account_id': account_id,
                    'normalized_account_id': normalize_contact(field, account_id),
                })
        return cleaned_data


class HireOrderVacancyForm(sform.SForm):

    name = sform.CharField(max_length=255, state=sform.REQUIRED)
    hiring_manager = UserSuggestField(state=sform.REQUIRED)
    department = DepartmentSuggestField(
        queryset=Department.objects.in_tree(settings.OUTSTAFF_DEPARTMENT_ID),
        to_field_name='url',
        state=sform.REQUIRED,
    )
    profession = ProfessionSuggestField(state=sform.REQUIRED)
    abc_services = ServiceMultipleSuggestField(to_field_name='slug', state=sform.REQUIRED)
    reason = sform.CharField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['profession'].queryset = Profession.active.exclude(
            professional_sphere_id__in=fetch_comma_separated_integers(
                config.HIRE_ORDER_DISABLED_PROF_SPHERE_IDS,
            ),
        )

    def clean(self):
        cleaned_data = super().clean()
        cleaned_data['type'] = VACANCY_TYPES.autohire
        return cleaned_data


class HireOrderOfferForm(ConditionallyRequiredFieldsMixin, sform.SForm):

    form_type = sform.ChoiceField(offer_choices.FORM_TYPES, state=sform.REQUIRED)
    org = sform.ModelChoiceField(
        queryset=Organization.objects.alive(),
        label_extractor=str,
        state=sform.REQUIRED,
    )
    position = PositionSuggestField(state=sform.REQUIRED)
    work_place = sform.ChoiceField(offer_choices.WORK_PLACES, state=sform.REQUIRED)
    office = sform.ModelChoiceField(
        queryset=Office.objects.alive().exclude_groups(),
        label_extractor=str,
    )
    homeworker_location = sform.CharField(max_length=255)
    join_at = sform.DateField(state=sform.REQUIRED)
    grade = PositiveIntegerField(max_value=100, state=sform.REQUIRED)
    payment_type = sform.ChoiceField(offer_choices.PAYMENT_TYPES, state=sform.REQUIRED)
    hourly_rate = PositiveDecimalField(
        max_digits=10,
        decimal_places=2,
    )
    salary = PositiveDecimalField(
        max_digits=10,
        decimal_places=2,
        state=sform.REQUIRED,
    )
    payment_currency = CurrencySuggestField(to_field_name='code', state=sform.REQUIRED)
    employment_type = sform.ChoiceField(offer_choices.EMPLOYMENT_TYPES, state=sform.REQUIRED)
    work_hours_weekly = PositiveIntegerField(max_value=168)
    is_main_work_place = sform.NullBooleanField(state=sform.REQUIRED)
    contract_type = sform.ChoiceField(offer_choices.OFFER_FORM_CONTRACT_TYPES, state=sform.REQUIRED)
    contract_term = PositiveIntegerField(max_value=1200)
    contract_term_date = sform.DateField()
    probation_period_type = sform.ChoiceField(
        choices=offer_choices.PROBATION_PERIOD_TYPES,
        state=sform.REQUIRED,
    )
    internet_compensation_amount = PositiveDecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
    )
    professional_level = sform.ChoiceField(offer_choices.PROFESSIONAL_LEVELS, state=sform.REQUIRED)
    hardware_profile_type = sform.ChoiceField(
        choices=offer_choices.HARDWARE_PROFILE_TYPES,
        default=offer_choices.HARDWARE_PROFILE_TYPES.no,
        state=sform.REQUIRED,
    )
    is_confirmed_by_boss = sform.BooleanField(default=True)
    is_internal_phone_needed = sform.NullBooleanField(default=False)
    is_sip_redirect_needed = sform.NullBooleanField(default=False)

    CONDITIONALLY_REQUIRED = {
        'homeworker_location': ('work_place', offer_choices.WORK_PLACES.home),
        'office': ('work_place', offer_choices.WORK_PLACES.office),
        'hourly_rate': ('payment_type', offer_choices.PAYMENT_TYPES.hourly),
        'contract_term': ('contract_type', offer_choices.OFFER_FORM_CONTRACT_TYPES.fixed_term),
        'contract_term_date': (
            'contract_type',
            offer_choices.OFFER_FORM_CONTRACT_TYPES.fixed_term_date,
        ),
        'work_hours_weekly': ('employment_type', offer_choices.EMPLOYMENT_TYPES.partial),
    }

    def clean(self):
        cleaned_data = super().clean()

        probation_period_type = cleaned_data.pop(
            'probation_period_type',
            offer_choices.PROBATION_PERIOD_TYPES.no,
        )
        cleaned_data['probation_period'], cleaned_data['probation_period_unit'] = (
            offer_choices.PROBATION_PERIOD_TYPE_TO_UNITS[probation_period_type]
        )

        work_place = cleaned_data.get('work_place')
        if work_place == offer_choices.WORK_PLACES.home:
            cleaned_data['office'] = Office.objects.get(pk=settings.OFFICE_HOMEWORKER_ID)

        employment_type = cleaned_data.get('employment_type')
        if employment_type == offer_choices.EMPLOYMENT_TYPES.full:
            cleaned_data['work_hours_weekly'] = 40

        position = cleaned_data.get('position')
        if position:
            cleaned_data['staff_position_name'] = position.name_ru

        return cleaned_data


class HireOrderMailForm(sform.SForm):
    """
    Дополнительные данные, нужные непосредственно для отправки оффера/анкеты КИ
    """
    sender = sform.EmailField()
    offer_message = sform.CharField()
    offer_receiver = sform.EmailField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['offer_receiver'].default = self.base_initial.get('offer_receiver')


class HireOrderForm(sform.SForm):

    recruiter = UserSuggestField(state=sform.REQUIRED)
    application_message = sform.CharField()
    force_verification_sending = sform.BooleanField()
    autofill_offer = sform.BooleanField()
    candidate = sform.FieldsetField(HireOrderCandidateForm, state=sform.REQUIRED)
    vacancy = sform.FieldsetField(HireOrderVacancyForm, state=sform.REQUIRED)
    offer = sform.FieldsetField(HireOrderOfferForm, state=sform.REQUIRED)
    mail = sform.FieldsetField(HireOrderMailForm)

    VALIDATE_ONLY_FIELDS = (
        'application_message',
        'force_verification_sending',
        'autofill_offer',
        'candidate',
        'vacancy',
        'offer',
        'mail',
    )

    def clean_recruiter(self, recruiter):
        validate_recruiters([recruiter])
        return recruiter

    def clean(self):
        cleaned_data = super().clean()
        cleaned_data['raw_data'] = self.data
        json_data = json.dumps(self.data, sort_keys=True, ensure_ascii=False)
        cleaned_data['sha1'] = hashlib.sha1(json_data.encode('utf-8')).hexdigest()
        for field in self.VALIDATE_ONLY_FIELDS:
            cleaned_data.pop(field, None)
        return cleaned_data


class HireOrderListFilterForm(sform.SForm):

    modified__gte = sform.DateTimeField(
        input_formats=ISO_INPUT_FORMATS['DATETIME_INPUT_FORMATS'] + [
            '%Y-%m-%d %H:%M:%S%z',
            '%Y-%m-%d %H:%M:%S.%f%z',
            '%Y-%m-%d %H:%M%z',
            '%Y-%m-%dT%H:%M:%S%z',
            '%Y-%m-%dT%H:%M:%S.%f%z',
            '%Y-%m-%dT%H:%M%z',
        ],
    )
    candidate_id = sform.IntegerField()
