import json

from constance import config
from rest_framework import permissions, status

from intranet.femida.src.api.core.errors import format_non_form_message
from intranet.femida.src.api.core.views import ResponseError
from intranet.femida.src.applications.models import Application
from intranet.femida.src.candidates.models import Candidate
from intranet.femida.src.core.workflow import WorkflowError
from intranet.femida.src.hire_orders.choices import HIRE_ORDER_STATUSES
from intranet.femida.src.hire_orders.models import HireOrder
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.vacancies.choices import VACANCY_TYPES
from intranet.femida.src.vacancies.models import Vacancy


class HireOrderPermission(permissions.BasePermission):

    def has_permission(self, request, view):
        return request.user.has_perm('permissions.can_use_hire_orders')

    def has_object_permission(self, request, view, obj):
        return (
            request.user.is_superuser
            or obj.created_by_id == request.user.id
        )


class ForbiddenForAutohire(permissions.BasePermission):
    """
    Право, разрешающее выполнение какого-либо действия,
    только если объект не проходит по автонайму.
    """
    def __init__(self, action_name=None):
        self.action_name = action_name

    def __call__(self):
        return self

    def has_object_permission(self, request, view, obj):
        action_name = self.action_name or view.actionlog_name
        is_forbidden_action = action_name in json.loads(config.AUTOHIRE_FORBIDDEN_ACTIONS)
        if is_forbidden_action:
            self._check_for_autohire(obj)
        return True

    def _check_for_autohire(self, obj):
        """
        Проверяет, не проходит ли `obj` по процессу автонайма.
        Если проходит, не просто отдаёт False, а выкидывает исключение.
        Сделано так для того, чтобы кинуть осмысленный код ошибки.
        """
        is_autohire = False
        if isinstance(obj, Vacancy):
            is_autohire = obj.type == VACANCY_TYPES.autohire
        elif isinstance(obj, (Application, Offer)):
            is_autohire = obj.vacancy.type == VACANCY_TYPES.autohire
        elif isinstance(obj, Candidate):
            is_autohire = (
                HireOrder.objects
                .exclude(status=HIRE_ORDER_STATUSES.closed)
                .filter(candidate=obj)
                .exists()
            )

        if is_autohire:
            error = WorkflowError('forbidden_for_autohire')
            data = format_non_form_message(error)
            raise ResponseError(data=data, status=status.HTTP_403_FORBIDDEN)
