from rest_framework import serializers

from intranet.femida.src.api.core.serializers import AwareSerializerMixin
from intranet.femida.src.hire_orders.models import HireOrder, HireOrderHistory
from intranet.femida.src.offers.controllers import OfferCtl
from intranet.femida.src.offers.helpers import get_candidate_alive_verifications_prefetch


class HireOrderVerificationField(serializers.ReadOnlyField):

    def __init__(self, source):
        super().__init__(source='candidate.alive_verifications')
        self._verification_attr = source

    def to_representation(self, value):
        return getattr(value[0], self._verification_attr, None) if value else None


class HireOrderDetailSerializer(AwareSerializerMixin, serializers.ModelSerializer):

    preprofile_id = serializers.ReadOnlyField(source='offer.newhire_id', default=None)
    tracker_job_key = serializers.ReadOnlyField(source='vacancy.startrek_key', default=None)
    tracker_hr_key = serializers.ReadOnlyField(source='offer.startrek_hr_key', default=None)
    tracker_hdrfs_key = serializers.ReadOnlyField(source='offer.startrek_hdrfs_key', default=None)
    verification_link = HireOrderVerificationField('link')
    verification_status = HireOrderVerificationField('status')
    verification_resolution = HireOrderVerificationField('resolution')
    hire_link = serializers.SerializerMethodField()
    join_at = serializers.DateField(source='offer.join_at', default=None)

    def get_hire_link(self, obj):
        if not obj.offer:
            return None
        return OfferCtl(obj.offer).link.url

    class Meta:
        model = HireOrder
        fields = (
            'uuid',
            'created',
            'modified',
            'status',
            'resolution',
            'resolution_description',
            'candidate_id',
            'vacancy_id',
            'offer_id',
            'preprofile_id',
            'tracker_job_key',
            'tracker_hr_key',
            'tracker_hdrfs_key',
            'verification_link',
            'verification_status',
            'verification_resolution',
            'hire_link',
            'join_at',
        )

        select_related_map = {
            'preprofile_id': ('offer',),
            'tracker_job_key': ('vacancy',),
        }

        prefetch_related_map = {
            'verification_link': (get_candidate_alive_verifications_prefetch(),),
        }


class HireOrderHistorySerializer(serializers.ModelSerializer):

    class Meta:
        model = HireOrderHistory
        fields = (
            'id',
            'status',
            'resolution',
            'changed_at',
        )
