import waffle

from django.db.models import QuerySet
from django.forms import BooleanField
from django.utils.decorators import method_decorator
from django.utils.functional import cached_property
from rest_framework import status
from rest_framework.parsers import JSONParser
from rest_framework.response import Response

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.api.core.errors import locked
from intranet.femida.src.api.core.pagination import CursorPagination
from intranet.femida.src.api.core.permissions import ServicePermission
from intranet.femida.src.api.core.views import BaseView, WorkflowView
from intranet.femida.src.api.hire_orders import forms, permissions, serializers
from intranet.femida.src.core.shortcuts import get_object_or_40x
from intranet.femida.src.hire_orders.choices import HIRE_ORDER_STATUSES, HIRE_ORDER_RESOLUTIONS
from intranet.femida.src.hire_orders.models import HireOrder, HireOrderHistory
from intranet.femida.src.hire_orders.workflow import HireOrderWorkflow
from intranet.femida.src.utils.table_flow import TableFlowAPI


class HireOrderHistoryPagination(CursorPagination):

    ordering = '-changed_at'
    page_size = 10
    max_page_size = 100


class HireOrderBaseView(BaseView):

    model_class = HireOrder
    detail_serializer_class = serializers.HireOrderDetailSerializer
    permission_classes = (
        ServicePermission('permissions.can_use_hire_orders'),
        permissions.HireOrderPermission,
    )


class HireOrderPagination(CursorPagination):

    page_size = 100
    max_page_size = 1000


class HireOrderListCreateView(HireOrderBaseView):

    validator_class = forms.HireOrderForm
    list_item_serializer_class = serializers.HireOrderDetailSerializer
    pagination_class = HireOrderPagination
    parser_classes = (JSONParser,)

    @action_logged('hire_order_create')
    def post(self, request, *args, **kwargs):
        use_table_flow = BooleanField().to_python(request.data.get('use_table_flow'))

        table_flow_data = {}
        if use_table_flow:
            table_flow_data = self.get_table_flow_data(request.data) or {}
            self.update_offer_params(request.data, table_flow_data)

        validator = self.get_validator_object(request.data)
        is_valid = validator.is_valid()
        cleaned_data = validator.cleaned_data
        cleaned_data['created_by'] = request.user
        cleaned_data['table_flow_data'] = table_flow_data

        if is_valid:
            instance = self.find_by_sha1(cleaned_data['sha1'])
            status_code = status.HTTP_201_CREATED
            if not instance:
                instance = self.perform_create(cleaned_data)
            elif waffle.switch_is_active('enable_409_on_hire_order_conflict'):
                status_code = status.HTTP_409_CONFLICT
            serializer = self.get_detail_serializer_object(instance)
            return Response(serializer.data, status=status_code)
        else:
            cleaned_data['status'] = HIRE_ORDER_STATUSES.closed
            cleaned_data['resolution'] = HIRE_ORDER_RESOLUTIONS.incorrect
            instance = self.model_class.objects.create(**cleaned_data)
            data = self._get_errors_data(validator)
            data['uuid'] = str(instance.uuid)
            return Response(data, status=status.HTTP_400_BAD_REQUEST)

    def get_table_flow_data(self, data):
        department = data.get('vacancy', {}).get('department')
        if department:
            return TableFlowAPI.get_hire_order_offer_params(department)

    def update_offer_params(self, data, table_flow_data):
        offer_data = data.get('offer', {})
        offer_data.update(table_flow_data)
        data['offer'] = offer_data

    def find_by_sha1(self, sha1):
        return (
            self.model_class.objects
            .filter(created_by=self.request.user, sha1=sha1)
            .exclude(status=HIRE_ORDER_STATUSES.closed)
            .first()
        )

    def perform_create(self, data):
        instance = self.model_class.objects.create(**data)
        workflow = HireOrderWorkflow(instance, user=None)
        workflow.delay('prepare_candidate')
        return instance

    def filter_queryset(self, queryset: 'QuerySet[HireOrder]'):

        filter_form = forms.HireOrderListFilterForm(self.request.query_params)
        self.validate(filter_form)
        filter_params = filter_form.cleaned_data

        modified__gte = filter_params.get('modified__gte')
        candidate_id = filter_params.get('candidate_id')

        if not self.request.user.is_superuser:
            queryset = queryset.filter(created_by=self.request.user)
        if modified__gte is not None:
            queryset = queryset.filter(modified__gte=modified__gte)
        if candidate_id is not None:
            queryset = queryset.filter(candidate_id=candidate_id)

        return queryset

    def get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)


class HireOrderDetailView(HireOrderBaseView):

    def get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, *kwargs)


class HireOrderHistoryListView(BaseView):

    model_class = HireOrderHistory
    list_item_serializer_class = serializers.HireOrderHistorySerializer
    permission_classes = (permissions.HireOrderPermission,)
    pagination_class = HireOrderHistoryPagination

    @cached_property
    def hire_order(self):
        instance = get_object_or_40x(HireOrder, uuid=self.kwargs['uuid'])
        self.check_object_permissions(self.request, instance)
        return instance

    def filter_queryset(self, queryset):
        return queryset.filter(hire_order=self.hire_order).order_by('-changed_at')

    def get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)


@method_decorator(locked, name='dispatch')
class HireOrderCancelView(WorkflowView):

    action_name = 'cancel'
    actionlog_name = 'hire_order_cancel'
    permission_classes = (permissions.HireOrderPermission,)
    workflow_class = HireOrderWorkflow
    detail_serializer_class = serializers.HireOrderDetailSerializer

    def get_queryset(self):
        return HireOrder.objects.select_for_update()
