import sform
import waffle

from django.conf import settings
from django.contrib.auth import get_user_model
from django.core.exceptions import ValidationError
from django.core.validators import MinValueValidator
from django.forms.models import ModelChoiceField as DjangoModelChoiceField
from django.template import Template, Context
from django.utils import timezone

from intranet.femida.src.api.core.fetchers import query_params_data_getter
from intranet.femida.src.api.core.forms import (
    ApplicationChoiceField,
    AttachmentMultipleSuggestField,
    BaseMetaForm,
    CandidateSuggestField,
    EMPTY_LABEL,
    ExtendedChoiceField,
    PresetSuggestField,
    TimeZoneChoiceField,
    UserChoiceField,
    UserSuggestField,
    UserMultipleSuggestField,
)
from intranet.femida.src.applications.helpers import active_applications_query
from intranet.femida.src.calendar import get_event
from intranet.femida.src.calendar.helpers import get_event_id_by_url
from intranet.femida.src.candidates.choices import CANDIDATE_STATUSES
from intranet.femida.src.candidates.helpers import get_main_email
from intranet.femida.src.communications.choices import (
    MESSAGE_TEMPLATE_TYPES,
    MESSAGE_TYPES,
    MESSAGE_STATUSES,
)
from intranet.femida.src.communications.models import MessageTemplate
from intranet.femida.src.core.choices import get_partial_choices
from intranet.femida.src.core.exceptions import SimpleValidationError
from intranet.femida.src.interviews import choices
from intranet.femida.src.interviews.helpers import get_enabled_aa_types, get_aa_types_count
from intranet.femida.src.interviews.models import Application, Interview
from intranet.femida.src.staff.models import Office
from intranet.femida.src.users.utils import filter_users_by_aa_type


User = get_user_model()


class DRFStyleErrorsMixin:

    _error_messages = {}

    def fail(self, code):
        raise ValidationError(
            message=self._error_messages.get(code, code),
            code=code,
        )


def _get_enabled_aa_types():
    return get_partial_choices(
        choices.AA_TYPES,
        *get_enabled_aa_types()
    )


class InterviewBaseForm(DRFStyleErrorsMixin, sform.SForm):

    _error_messages = {
        'wrong_event_url': 'Неправильный URL события в Календаре',
        'aa_conflict': 'АА-секция может быть проведена только АА-собеседующим',
        'invalid_choice': DjangoModelChoiceField.default_error_messages['invalid_choice'],
    }

    type = sform.ChoiceField(
        choices=choices.INTERVIEW_TYPES,
        state=sform.REQUIRED,
        empty_label=EMPTY_LABEL,
    )
    # Порядок event_id и event_url важен для валидации
    event_id = sform.IntegerField()
    event_url = sform.CharField()

    event_instance_start_ts = sform.CharField()
    event_start_ts = sform.DateTimeField()
    event_resources = sform.GridField(sform.CharField())
    event_others_can_view = sform.BooleanField()
    event_conference_url = sform.CharField()
    is_new_form = sform.BooleanField()

    section = sform.CharField(max_length=255, state=sform.REQUIRED)
    interviewer = UserSuggestField(state=sform.REQUIRED)
    optional_participant = UserSuggestField()
    aa_type = ExtendedChoiceField(
        choices=choices.AA_TYPES,
    )
    is_code = sform.BooleanField()
    application = ApplicationChoiceField(
        queryset=Application.unsafe.none(),
        empty_label=EMPTY_LABEL,
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        candidate = self.context.get('candidate')
        assert candidate is not None
        self.fields['application'].queryset = (
            Application.objects
            .filter(active_applications_query)
            .filter(candidate_id=candidate.id)
        )
        # Нельзя внести в параметры поля класса, т.к. используется база на этапе запуска,
        # что ломает создание миграций
        self.fields['aa_type'].choices = _get_enabled_aa_types()
        self.fields['aa_type'].extra = {
            'robot': settings.AA_TYPES_ROBOTS, 'count': get_aa_types_count()
        }

    def _get_event_id(self):
        return self.cleaned_data.get('event_id', None)

    def _get_interview_type(self):
        return self.cleaned_data.get('type')

    def get_field_state(self, name):
        is_required = False
        interview_type = self._get_interview_type()
        if name == 'application':
            is_required = interview_type in (
                choices.INTERVIEW_TYPES.screening,
                choices.INTERVIEW_TYPES.regular,
                choices.INTERVIEW_TYPES.final,
            )
        if name == 'event_url':
            event_id = self._get_event_id()
            is_required = interview_type in choices.INTERVIEW_GRADABLE_TYPES and not event_id
        if is_required:
            return sform.REQUIRED
        return super().get_field_state(name)

    def _extract_event_id(self, cleaned_data):
        event_id = cleaned_data.get('event_id', None)
        event_url = cleaned_data.pop('event_url', None)

        if event_id:
            return event_id

        if event_url:
            event_id = get_event_id_by_url(event_url)
            if event_id:
                return event_id
            self.fail('wrong_event_url')

        return None

    # TODO: заменить на cleaned_data.pop, когда все поля будут использоваться
    event_props = [
        'event_start_ts',
        'event_resources',
        'event_others_can_view',
        'event_conference_url',
    ]

    def _clean_event_data(self, cleaned_data):
        for prop in self.event_props:
            cleaned_data.pop(prop, None)

    def _process_event(self, cleaned_data):
        event_id = self._extract_event_id(cleaned_data)
        if event_id:
            user = self.context.get('user')
            user_ticket = self.context.get('user_ticket')
            if user and user_ticket:
                event_instance_start_ts = cleaned_data.get('event_instance_start_ts', None)
                event = get_event(
                    event_id=event_id,
                    uid=user.uid,
                    user_ticket=user_ticket,
                    instance_start_ts=event_instance_start_ts,
                )
            else:
                event = None
            if event:
                event.others_can_view = cleaned_data.get(
                    'event_others_can_view', event.others_can_view
                )
                event.conference_url = cleaned_data.get(
                    'event_conference_url', event.conference_url
                )
                event_resources = cleaned_data.get('event_resources')
                event_instance_start_ts = cleaned_data.get('event_instance_start_ts', None)
                if event_instance_start_ts:
                    event.instance_start_ts = event_instance_start_ts
                if event_resources:
                    event.resources = event_resources
                cleaned_data['event'] = event
                cleaned_data['event_id'] = event_id
                cleaned_data['event_start_time'] = event.start_time
        else:
            cleaned_data['event'] = None
            cleaned_data['event_id'] = None
            cleaned_data['event_start_time'] = None
            cleaned_data['event_instance_start_ts'] = None

        self._clean_event_data(cleaned_data)

    def clean(self):
        cleaned_data = super().clean()

        self._process_event(cleaned_data)

        interview_type = self._get_interview_type()
        interviewer = cleaned_data.get('interviewer')
        is_aa = interview_type == choices.INTERVIEW_TYPES.aa
        aa_type = cleaned_data.get('aa_type')

        if is_aa and interviewer and not interviewer.is_aa:
            self.fail('aa_conflict')

        if is_aa and interviewer and not aa_type:
            cleaned_data['aa_type'] = interviewer.aa_type

        return cleaned_data


class InterviewCreateForm(InterviewBaseForm):

    candidate = CandidateSuggestField(state=sform.REQUIRED)
    preset = PresetSuggestField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        candidate = self.context.get('candidate')
        assert candidate is not None
        self.fields['candidate'].default = candidate.id

    def clean_application(self, application):
        candidate = self.cleaned_data.get('candidate')
        if application and candidate and application.candidate_id != candidate.id:
            self.fail('invalid_choice')
        return application

    def clean_candidate(self, candidate):
        if candidate.status == CANDIDATE_STATUSES.closed:
            self.fail('candidate_closed')
        return candidate


class InterviewUpdateForm(InterviewBaseForm):

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields.pop('type')

    def _get_interview_type(self):
        interview = self.context.get('interview')
        if interview:
            return interview.type


class AAInterviewUpdateForm(InterviewUpdateForm):

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields.pop('application')


class HRScreeningUpdateForm(sform.SForm):

    section = sform.CharField(max_length=255, state=sform.REQUIRED)
    interviewer = UserSuggestField(state=sform.REQUIRED)


# TODO: удалить после перехода на InterviewCommentForm
class InterviewCommonUpdateForm(InterviewUpdateForm):

    comment = sform.CharField()
    resolution = sform.ChoiceField(choices.INTERVIEW_RESOLUTIONS)


class InterviewWorkflowForm(DRFStyleErrorsMixin, sform.SForm):

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.interview = self.context.get('interview')


class InterviewCommentForm(InterviewWorkflowForm):
    """
    Добавление комментария в секцию
    """
    comment = sform.CharField()


class InterviewFinishForm(InterviewWorkflowForm):

    _error_messages = {
        'missing_grade': 'Оцените собеседование',
        'missing_resolution': 'Оцените собеседование',
        'empty_interview_comment': 'Общий комментарий к секции не может быть пустым',
        'assignments_without_grade_exist': 'Поставьте оценку всем добавленным задачам',
        'missing_yandex_grade': 'Оцените собеседование по новой шкале',
        'missing_scale': 'Оцените, с какой шкалой было удобнее работать',
    }

    def clean(self):
        for condition, error in self._get_checks():
            if condition:
                self.fail(error)
        return super().clean()

    def _get_checks(self):
        i = self.interview
        yield i.is_gradable and i.grade is None, 'missing_grade'
        yield not i.is_gradable and not i.resolution, 'missing_resolution'
        yield not i.comment, 'empty_interview_comment'
        yield (
            i.is_gradable and i.yandex_grade is None and self._require_yandex_grade,
            'missing_yandex_grade',
        )
        yield i.is_gradable and not i.scale and self._require_scale, 'missing_scale'
        yield i.assignments.filter(grade__isnull=True).exists(), 'assignments_without_grade_exist'

    @property
    def _require_yandex_grade(self):
        return waffle.switch_is_active('require_interview_yandex_grade')

    @property
    def _require_scale(self):
        return waffle.switch_is_active('require_interview_scale')


class InterviewCancelForm(sform.SForm):

    cancel_reason = sform.CharField()


class InterviewEstimateForm(InterviewWorkflowForm):

    _error_messages = {
        'already_estimated': 'Нельзя повторно оценить собеседование'
    }

    grade = sform.ChoiceField(choices=Interview.GRADES, state=sform.REQUIRED)

    def clean_grade(self, grade):
        return int(grade) if grade else None

    def clean(self):
        if self.interview.grade is not None:
            self.fail('already_estimated')
        return super().clean()


class InterviewReassignForm(InterviewWorkflowForm):

    _error_messages = {
        'interviewer_should_be_aa': 'Собеседующий должен быть AA-интервьюером',
    }

    interviewer = UserSuggestField(state=sform.REQUIRED)
    reason = sform.CharField()

    def clean(self):
        cleaned_data = super().clean()
        if self.interview.is_aa and not cleaned_data['interviewer'].is_aa:
            self.fail('interviewer_should_be_aa')

        return cleaned_data


def _extract_name_and_login(user):
    return ('%s %s, %s@' % (user.first_name, user.last_name, user.username)).strip(' ,')


class AAInterviewerReassignForm(InterviewWorkflowForm):

    interviewer = UserChoiceField(
        queryset=User.objects.order_by('username'),
        label_extractor=_extract_name_and_login,
        state=sform.REQUIRED,
    )
    reason = sform.CharField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        aa_type = self.interview.aa_type
        qs = self.fields['interviewer'].queryset
        self.fields['interviewer'].queryset = filter_users_by_aa_type(qs, aa_type)


class InterviewChangeResolutionForm(InterviewWorkflowForm):

    resolution = sform.ChoiceField(choices=choices.INTERVIEW_RESOLUTIONS, state=sform.REQUIRED)


class InterviewRenameForm(InterviewWorkflowForm):

    section = sform.CharField(max_length=255, state=sform.REQUIRED)


class InterviewChangeGradeForm(InterviewWorkflowForm):

    yandex_grade = sform.ChoiceField(choices=choices.INTERVIEW_YANDEX_GRADES, state=sform.REQUIRED)

    def clean_yandex_grade(self, value):
        return int(value) if value else None


class InterviewChooseScaleForm(InterviewWorkflowForm):

    scale = sform.ChoiceField(choices=choices.INTERVIEW_SCALES, state=sform.REQUIRED)


class InterviewListFilterFrontendForm(sform.SForm):
    """
    Базовая форма, которая используется в ручке формы фильтра списка секций
    """
    default_getter = query_params_data_getter

    interviewer = UserSuggestField()
    types = sform.MultipleChoiceField(choices=choices.INTERVIEW_TYPES)
    state = sform.ChoiceField(
        choices=choices.INTERVIEW_ALIVE_STATES,
        empty_label=EMPTY_LABEL,
    )
    resolution = sform.ChoiceField(
        choices=choices.INTERVIEW_RESOLUTIONS,
        empty_label=EMPTY_LABEL,
    )
    grades = sform.MultipleChoiceField(choices=choices.INTERVIEW_GRADES)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        user = self.context.get('user')
        if user:
            self.fields['interviewer'].default = user.username


class InterviewListFilterForm(InterviewListFilterFrontendForm):
    """
    Форма, по которой фактически происходит фильтрация списка секций
    """
    only_active = sform.BooleanField(default=False)


class InterviewAchievementForm(sform.SForm):

    interviewer = UserSuggestField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        user = self.context.get('user')
        if user:
            self.fields['interviewer'].default = user.username


class InterviewForRoundCreateForm(sform.SForm):

    name = sform.CharField(max_length=255, state=sform.REQUIRED)
    aa_type = sform.ChoiceField(choices=choices.AA_TYPES)
    application = ApplicationChoiceField(
        queryset=Application.unsafe.none(),
        empty_label=EMPTY_LABEL,
    )
    is_code = sform.BooleanField()
    preset = PresetSuggestField()
    potential_interviewers = UserMultipleSuggestField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['application'].queryset = self.base_initial['_active_applications_qs']
        # TODO: заменить на _get_enabled_aa_types
        self.fields['aa_type'].choices = get_partial_choices(
            choices.AA_TYPES,
            *get_enabled_aa_types()
        )

    def clean(self):
        super().clean()
        self.cleaned_data['section'] = self.cleaned_data.pop('name', '')
        return self.cleaned_data

    def get_field_state(self, name):
        if name in ('potential_interviewers', 'application'):
            return sform.READONLY if self.cleaned_data.get('aa_type') else sform.REQUIRED
        return super().get_field_state(name)


class TimeSlotForm(sform.SForm):

    start = sform.DateTimeField(state=sform.REQUIRED)
    end = sform.DateTimeField(
        state=sform.REQUIRED,
        validators=[MinValueValidator(timezone.now)],
    )


class InterviewRoundCreateForm(BaseMetaForm):

    type = sform.ChoiceField(
        choices=choices.INTERVIEW_ROUND_TYPES,
        empty_label=EMPTY_LABEL,
        state=sform.REQUIRED,
    )
    is_any_time = sform.BooleanField()
    is_strict_order = sform.BooleanField()
    lunch_duration = sform.ChoiceField(
        choices=choices.INTERVIEW_ROUND_LUNCH_DURATIONS,
        default=choices.INTERVIEW_ROUND_LUNCH_DURATIONS.minutes_30,
        state=sform.REQUIRED,
    )
    office = sform.ModelChoiceField(
        queryset=(
            Office.objects
            .alive()
            .order_by('name_ru')
        ),
        empty_label=EMPTY_LABEL,
        label_extractor=lambda x: x.name,
        state=sform.REQUIRED,
    )
    time_slots = sform.GridField(sform.FieldsetField(TimeSlotForm))
    interviews = sform.GridField(
        field_instance=sform.FieldsetField(InterviewForRoundCreateForm),
        state=sform.REQUIRED,
    )
    comment = sform.CharField()

    need_notify_candidate = sform.BooleanField(default=True)
    email = sform.EmailField(max_length=255)
    subject = sform.CharField(max_length=255)
    text = sform.CharField()
    attachments = AttachmentMultipleSuggestField()
    timezone = TimeZoneChoiceField(default='Europe/Moscow')
    ordering = sform.ChoiceField(
        choices=choices.INTERVIEW_ROUND_ORDERINGS,
        default=choices.INTERVIEW_ROUND_ORDERINGS.any,
        state=sform.REQUIRED,
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['office'].default = (
            Office.objects.filter(id=settings.REMOTE_OFFICE_ID).values_list('id', flat=True).first()
        )
        email = get_main_email(self.base_initial['_candidate'])
        if email:
            self.fields['email'].default = email

    def clean_text(self, text):
        if text:
            try:
                # Не допускаем ошибок вида: '{}', '{time}', '{ interview_datetime }', '{'.
                # Удаление параметра допускаем.
                text.format(interview_datetime='')
            except (KeyError, IndexError, ValueError):
                raise SimpleValidationError('invalid_parameter')
        return text

    def clean(self):
        super().clean()
        self.cleaned_data.pop('timezone', None)
        if self.cleaned_data.pop('is_any_time', False):
            self.cleaned_data['time_slots'] = []

        ordering = self.cleaned_data.pop('ordering', None)
        if ordering:
            self._process_ordering(ordering)

        self.cleaned_data['message'] = {
            'email': self.cleaned_data.pop('email', ''),
            'subject': self.cleaned_data.pop('subject', ''),
            'text': self.cleaned_data.pop('text', ''),
            'attachments': self.cleaned_data.pop('attachments', []),
            'candidate': self.base_initial['_candidate'],
            'type': MESSAGE_TYPES.outcoming,
            'status': MESSAGE_STATUSES.draft,
        }
        return self.cleaned_data

    def get_field_state(self, name):
        if name == 'time_slots' and not self.cleaned_data.get('is_any_time'):
            return sform.REQUIRED
        if name in ('email', 'subject', 'text') and self.cleaned_data.get('need_notify_candidate'):
            return sform.REQUIRED
        return super().get_field_state(name)

    def meta_as_dict(self):
        templates, signatures = self._get_templates_and_signatures()

        template_context = Context({'candidate_name': self.base_initial['_candidate'].first_name})
        for template in templates:
            template['text'] = Template(template['text']).render(template_context)

        signature_context = Context({'user': self.base_initial['_user']})
        for signature in signatures:
            signature['text'] = Template(signature['text']).render(signature_context)
        return {
            'templates': templates,
            'signatures': signatures,
        }

    def _get_templates_and_signatures(self):
        message_templates = (
            MessageTemplate.objects
            .filter(
                type__in=(MESSAGE_TEMPLATE_TYPES.interview, MESSAGE_TEMPLATE_TYPES.signature),
                is_active=True,
            )
            .order_by('type', 'category_name', 'name')
            .values('type', 'category_name', 'name', 'subject', 'text')
        )
        templates = []
        signatures = []
        for template in message_templates:
            if template.pop('type') == MESSAGE_TEMPLATE_TYPES.interview:
                templates.append(template)
            else:
                template.pop('category_name')
                template.pop('subject')
                signatures.append(template)
        return templates, signatures

    def _process_ordering(self, ordering):
        if ordering == choices.INTERVIEW_ROUND_ORDERINGS.strict:
            self.cleaned_data['is_strict_order'] = True
        if ordering == choices.INTERVIEW_ROUND_ORDERINGS.aa_last:
            self.cleaned_data['comment'] += '\n!!АА секцию последней!!'


class InterviewSendToReviewForm(sform.SForm):

    comment = sform.CharField(state=sform.REQUIRED)
