from django.conf import settings
from django.db.models import Prefetch
from rest_framework import serializers

from intranet.femida.src.api.assignments.serializers import (
    AssignmentSerializer,
    AssignmentLiteSerializer,
)
from intranet.femida.src.api.candidates.serializers import ConsiderationLiteSerializer
from intranet.femida.src.api.candidates.utils import (
    get_unique_serialized_attachments,
    get_unique_contacts,
)
from intranet.femida.src.api.core.fields import SafeWikiFormattedField
from intranet.femida.src.api.core.serializers import (
    AwareSerializerMixin,
    FemidaSerializer,
    WorkflowActionsField,
)
from intranet.femida.src.api.users.serializers import UserSerializer
from intranet.femida.src.api.vacancies.serializers import VacancyLiteSerializer
from intranet.femida.src.candidates.choices import CONTACT_TYPES
from intranet.femida.src.candidates.considerations.helpers import (
    add_items_counts_to_considerations_qs,
)
from intranet.femida.src.candidates.models import Candidate, Consideration
from intranet.femida.src.interviews.helpers import can_user_see_review_issue
from intranet.femida.src.interviews.models import Interview
from intranet.femida.src.interviews.workflow import InterviewWorkflow


def get_partial_serializer(serializer_class, fields, **kwargs):
    """
    Возвращает сериализатор, полученный из `serializer_class` путем исключения полей,
    не указанных в `fields`, и относящихся к ним _related_map'ов

    :param serializer_class: Урезаемый сериализатор. Должен быть унаследован от AwareSerializerMixin
    :param fields: Поля, которые будут оставлены в результирующем сериализаторе
    """
    return serializer_class(
        context={
            'fields': fields,
        },
        **kwargs
    )


class CandidateForInterviewSerializer(AwareSerializerMixin, FemidaSerializer):

    resume = serializers.SerializerMethodField()
    skype_contacts = serializers.SerializerMethodField()
    extended_status = serializers.SerializerMethodField()

    def get_resume(self, obj):
        attachments = get_unique_serialized_attachments(obj)
        return attachments[0] if attachments else None

    def get_skype_contacts(self, obj):
        contacts = get_unique_contacts(obj)
        return [c for c in contacts if c['type'] == CONTACT_TYPES.skype]

    def get_extended_status(self, obj):
        if 'extended_statuses' in self.root.context:
            return self.root.context['extended_statuses'].get(obj.id)

    class Meta:
        model = Candidate
        fields = (
            'id',
            'first_name',
            'middle_name',
            'last_name',
            'login',
            'resume',
            'skype_contacts',
            'extended_status',
        )
        prefetch_related_map = {
            'resume': (
                'candidate_attachments__attachment',
            ),
            'skype_contacts': (
                'contacts',
            ),
        }


class InterviewSerializer(AwareSerializerMixin, FemidaSerializer):

    actions = WorkflowActionsField(InterviewWorkflow)
    assignments = AssignmentLiteSerializer(many=True)
    created_by = UserSerializer()
    formatted_comment = SafeWikiFormattedField()
    grade_verbose = serializers.ReadOnlyField(source='get_grade_display')
    interviewer = UserSerializer()
    optional_participant = UserSerializer()
    result = serializers.SerializerMethodField()
    startrek_review_key = serializers.SerializerMethodField()
    preset_name = serializers.ReadOnlyField(source='preset.name', default=None)
    candidate = get_partial_serializer(
        serializer_class=CandidateForInterviewSerializer,
        fields=('id', 'first_name', 'middle_name', 'last_name', 'login'),
    )
    consideration = ConsiderationLiteSerializer()
    application = serializers.SerializerMethodField()
    vacancy = get_partial_serializer(
        serializer_class=VacancyLiteSerializer,
        fields=('id', 'name', 'status', 'is_available'),
        source='application.vacancy',
        default=None,
    )

    # TODO: Эти поля нужны только для магических ссылок
    candidate_name = serializers.ReadOnlyField(source='candidate.get_full_name')
    state_verbose = serializers.ReadOnlyField(source='get_state_display')

    def get_result(self, obj):
        if obj.grade:
            return 'hire {}'.format(obj.grade)
        elif obj.grade == 0:
            return 'nohire'
        else:
            return None

    def get_startrek_review_key(self, obj):
        user = self.root.context.get('user')
        if user and can_user_see_review_issue(user, obj):
            return obj.startrek_review_key

    def get_application(self, obj):
        if obj.application is not None:
            return {
                'id': obj.application_id,
                'status': obj.application.status,
                'resolution': obj.application.resolution,
            }

    class Meta:
        model = Interview
        fields = (
            'id',
            'type',
            'actions',
            'application',
            'assignments',
            'comment',
            'created',
            'created_by',
            'finished',
            'formatted_comment',
            'grade',
            'grade_verbose',
            'yandex_grade',
            'scale',
            'is_pro_level_scale',
            'interviewer',
            'optional_participant',
            'event_id',
            'result',
            'section',
            'state',
            'state_verbose',
            'candidate_name',
            'resolution',
            'candidate',
            'consideration',
            'vacancy',
            'aa_type',
            'is_code',
            'startrek_review_key',
            'event_start_time',
            'event_instance_start_ts',
            'preset_name',
        )
        select_related_map = {
            'vacancy': ('application__vacancy',),
            'created_by': ('created_by',),
            'interviewer': ('interviewer',),
            'preset': ('preset',),
            'optional_participant': ('optional_participant',),
            'candidate': ('candidate',),
            'candidate_name': ('candidate',),
        }
        prefetch_related_map = {
            'assignments': (
                'assignments__problem',
            ),
            'consideration': (
                Prefetch(
                    lookup='consideration',
                    queryset=add_items_counts_to_considerations_qs(
                        qs=Consideration.unsafe.all(),
                    ),
                ),
                'consideration__responsibles',

                # Note: Это нужно для actions, но если положить в правильное место,
                # работает долго, предположительно из-за того,
                # что сначала мы префетчим consideration__interviews,
                # а потом натыкаемся на prefetch consideration'а,
                # и все становится плохо
                'consideration__interviews',
            ),
        }


class InterviewCreateSerializer(InterviewSerializer):

    candidate = CandidateForInterviewSerializer()


class InterviewDetailSerializer(InterviewSerializer):

    assignments = AssignmentSerializer(many=True)
    candidate = CandidateForInterviewSerializer()

    class Meta(InterviewSerializer.Meta):
        prefetch_related_map = getattr(InterviewSerializer.Meta, 'prefetch_related_map', {}).copy()
        prefetch_related_map.update({
            'assignments': (
                'assignments',
                'assignments__problem__created_by',
                'assignments__problem__categories',
                'assignments__problem__fans',
            ),
        })


class InterviewsReportSerializer(serializers.ModelSerializer):

    url = serializers.SerializerMethodField()
    name = serializers.CharField(source='section')
    created = serializers.DateTimeField(format='%Y-%m-%d %H:%M:%S')
    finished = serializers.DateTimeField(format='%Y-%m-%d %H:%M:%S')
    event_start_time = serializers.DateTimeField(format='%Y-%m-%d %H:%M:%S')
    is_code = serializers.IntegerField()
    is_pro_level_scale = serializers.IntegerField()
    candidate_id = serializers.IntegerField(source='consideration__candidate_id')
    candidate_full_name = serializers.SerializerMethodField()
    candidate_login = serializers.CharField(source='consideration__candidate__login')

    vacancy_id = serializers.ReadOnlyField(source='application__vacancy_id')
    vacancy_name = serializers.ReadOnlyField(source='application__vacancy__name')
    vacancy_department = serializers.ReadOnlyField(source='application__vacancy__department__url')
    vacancy_cities = serializers.SerializerMethodField()
    vacancy_profession = serializers.ReadOnlyField(source='application__vacancy__profession__name')
    vacancy_pro_level_max = serializers.ReadOnlyField(source='application__vacancy__pro_level_max')

    interviewer = serializers.CharField(source='interviewer__username')
    recruiter = serializers.CharField(source='created_by__username')
    consideration_id = serializers.IntegerField()

    def get_vacancy_cities(self, obj):
        if 'vacancy_cities_map' in self.root.context:
            return ', '.join(
                self.root.context['vacancy_cities_map']
                .get(obj['application__vacancy_id'], [])
            )

    def get_candidate_full_name(self, obj):
        return '{} {}'.format(
            obj['consideration__candidate__first_name'],
            obj['consideration__candidate__last_name']
        )

    def get_url(self, obj):
        return '{protocol}://{host}/interviews/{interview_id}'.format(
            protocol=settings.FEMIDA_PROTOCOL,
            host=settings.FEMIDA_HOST,
            interview_id=obj['id'],
        )

    class Meta:
        model = Interview
        fields = (
            'id',
            'type',
            'url',
            'name',
            'created',
            'finished',
            'event_start_time',
            'state',
            'grade',
            'aa_type',
            'is_code',
            'is_pro_level_scale',
            'candidate_id',
            'candidate_full_name',
            'candidate_login',
            'vacancy_id',
            'vacancy_name',
            'vacancy_department',
            'vacancy_cities',
            'vacancy_profession',
            'vacancy_pro_level_max',
            'interviewer',
            'recruiter',
            'consideration_id',
        )


class InterviewsReportFilterForm(serializers.Serializer):

    interviewers = serializers.CharField(required=False)
    recruiters = serializers.CharField(required=False)
    states = serializers.CharField(required=False)
    date_from = serializers.DateField(required=False)
    date_to = serializers.DateField(required=False)
    departments = serializers.CharField(required=False)
    _fields = serializers.CharField(required=False)


class InterviewUpdateFormSerializer(serializers.ModelSerializer):

    interviewer = serializers.CharField(source='interviewer.username')
    application = serializers.IntegerField(source='application_id')

    class Meta:
        model = Interview
        fields = (
            'section',
            'interviewer',
            'application',
            'event_url',
            'aa_type',
            'is_code',
        )


class InterviewReassignFormSerializer(serializers.Serializer):

    interviewer = serializers.CharField(source='interviewer.username')


class InterviewChangeResolutionFormSerializer(serializers.Serializer):

    resolution = serializers.CharField()


class InterviewRenameFormSerializer(serializers.Serializer):

    section = serializers.CharField()


class InterviewChangeGradeFormSerializer(serializers.Serializer):

    yandex_grade = serializers.ReadOnlyField()


class InterviewChooseScaleFormSerializer(serializers.Serializer):

    scale = serializers.ReadOnlyField()
