from collections import defaultdict

from django.db.models import Q
from django.utils import timezone
from rest_framework import status
from rest_framework.exceptions import PermissionDenied
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.settings import api_settings
from rest_framework_csv.renderers import CSVRenderer

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.api.comments.views import CommentListCreateBaseView
from intranet.femida.src.api.core.forms import AddProblemsFromPresetForm
from intranet.femida.src.api.core.renderers import XLSRenderer
from intranet.femida.src.api.core.views import (
    BaseView,
    WorkflowView,
    BaseFormView,
    InstanceFormViewMixin,
    ResponseError,
    unfold_query_params,
)
from intranet.femida.src.api.candidates.mixins import CandidateHelperFormMixin
from intranet.femida.src.api.candidates.permissions import NoDuplicatesPermission
from intranet.femida.src.api.hire_orders.permissions import ForbiddenForAutohire
from intranet.femida.src.calendar import get_event
from intranet.femida.src.candidates.considerations.helpers import get_considerations_items_counts
from intranet.femida.src.candidates.controllers import get_candidates_extended_statuses
from intranet.femida.src.core.db import OrderExpression
from intranet.femida.src.core.renderers import UTF8JSONRenderer
from intranet.femida.src.core.shortcuts import get_object_or_40x
from intranet.femida.src.interviews.choices import INTERVIEW_STATES, INTERVIEW_TYPES
from intranet.femida.src.interviews.controllers import create_interview
from intranet.femida.src.interviews.helpers import add_calculated_resolution_to_qs
from intranet.femida.src.interviews.workflow import InterviewWorkflow
from intranet.femida.src.interviews.models import Interview
from intranet.femida.src.problems.helpers import get_active_complaints_counts, get_problems_stats
from intranet.femida.src.problems.utils import reorder_interview, get_problems_usage_in_context
from intranet.femida.src.staff.models import Department
from intranet.femida.src.vacancies.models import Vacancy, VacancyCity
from . import serializers, forms
from .permissions import (
    InterviewPermission,
    InterviewListCreatePermission,
    InterviewsReportPermission,
)


class InterviewListCreateView(CandidateHelperFormMixin, BaseView):

    model_class = Interview
    permission_classes = [
        IsAuthenticated,
        InterviewListCreatePermission,
        NoDuplicatesPermission,
        ForbiddenForAutohire('interview_create'),
    ]
    list_item_serializer_class = serializers.InterviewSerializer
    detail_serializer_class = serializers.InterviewCreateSerializer
    validator_class = forms.InterviewCreateForm

    def get_list_item_serializer_context(self):
        context = super().get_list_item_serializer_context()
        context['available_vacancy_ids'] = set(
            Vacancy.objects
            .filter(id__in=[i.application.vacancy_id for i in self.page if i.application])
            .values_list('id', flat=True)
        )
        return context

    def get_detail_serializer_context(self):
        consideration_ids = self.candidate.considerations.values('id')
        return {
            'considerations_items_counts': get_considerations_items_counts(consideration_ids),
            'extended_statuses': get_candidates_extended_statuses([self.candidate.id]),
        }

    def get(self, request, *args, **kwargs):
        """
        Список собеседований
        ---
        parameters:
            - name: interviewer
              description: >
                Логин собеседующего. В выборку попадут собеседования,
                актуальные для него
              required: false
              type: string
              paramType: query
        """
        return self.list(request, *args, **kwargs)

    @action_logged('interview_create')
    def post(self, request, *args, **kwargs):
        """
        Создание собеседования
        """
        response = self.create(request, *args, **kwargs)
        response.set_cookie('just_updated', 'true', max_age=5)
        return response

    def perform_create(self, data):
        return create_interview(created_by=self.request.user, **data)

    def get_validator_context(self):
        return {
            'candidate': self.candidate,
            'user_ticket': self.user_ticket,
        }

    def filter_queryset(self, queryset, *args, **kwargs):
        params = unfold_query_params(
            query_params=self.request.query_params,
            list_fields=('types', 'grades'),
        )
        filter_form = forms.InterviewListFilterForm(
            data=params,
            context={
                'user': self.request.user,
            },
        )
        self.validate(filter_form)
        filter_params = filter_form.cleaned_data

        # Если передан interviewer, ищем секции c этим собеседующим
        # Иначе ищем актуальные секции для текущего пользователя
        if filter_params['interviewer']:
            # По параметру interviewer фильтровать могут только
            # рекрутер, АА-бэкенд и пользователь с логином == interviewer
            if (not self.request.user.is_recruiter
                    and not self.request.user.is_aa_canonical
                    and self.request.user != filter_params['interviewer']):
                return queryset.none()

            # Если передан параметр interviewer, то возвращаем только
            # собеседования, актуальные для этого собеседующего
            queryset = queryset.filter(interviewer=filter_params['interviewer'])
        else:
            # Note: Оптимальнее фильтровать по id, т.к. здесь OR.
            # При таком подходе фильтрация идет по одной таблице,
            # а не по 2-м, как было при фильтрации по username.
            # PostgreSQL в таком случае выбирает более оптимальный план.
            user_id = self.request.user.id
            queryset = queryset.filter(Q(interviewer_id=user_id) | Q(created_by_id=user_id))

        active_states = [Interview.STATES.assigned, Interview.STATES.estimated]
        if filter_params['only_active']:
            today = timezone.now().date()
            queryset = queryset.filter(
                Q(state__in=active_states)
                | Q(state=Interview.STATES.finished, finished__gt=today)
            )
        else:
            queryset = queryset.filter(state__in=active_states + [Interview.STATES.finished])

        if filter_params['types']:
            queryset = queryset.filter(type__in=filter_params['types'])

        if filter_params['state']:
            queryset = queryset.filter(state=filter_params['state'])

        if filter_params['resolution']:
            queryset = add_calculated_resolution_to_qs(queryset)
            queryset = queryset.filter(calculated_resolution=filter_params['resolution'])

        if filter_params['grades']:
            queryset = queryset.filter(grade__in=filter_params['grades'])

        queryset = (
            queryset
            .annotate(state_order=OrderExpression('state', INTERVIEW_STATES))
            .order_by('state_order', '-finished', 'event_start_time')
        )
        queryset = queryset.filter(consideration__isnull=False)

        return queryset


class InterviewListFilterFormView(BaseFormView):

    model_class = Interview
    validator_class = forms.InterviewListFilterFrontendForm

    def get_initial_data(self):
        return unfold_query_params(
            query_params=self.request.query_params,
            list_fields=('types', 'grades'),
        )


class InterviewAchievementView(BaseView):

    model_class = Interview
    validator_class = forms.InterviewAchievementForm

    def filter_queryset(self, queryset):
        filter_form = self.get_validator_object(self.request.query_params)
        self.validate(filter_form)
        filter_params = filter_form.cleaned_data

        interviewer = filter_params['interviewer']
        if interviewer:
            is_filter_available = (
                self.request.user.is_recruiter
                or self.request.user == interviewer
            )
            if not is_filter_available:
                return queryset.none()
            queryset = queryset.filter(interviewer=interviewer)

        return queryset

    def get(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())
        achievement_interview_count = (
            queryset
            .gradable()
            .filter(state=INTERVIEW_STATES.finished)
            .count()
        )
        data = {
            'achievement_interview_count': achievement_interview_count,
        }
        return Response(data)


class InterviewsReportView(BaseView):

    model_class = Interview
    list_item_serializer_class = serializers.InterviewsReportSerializer
    permission_classes = [IsAuthenticated, InterviewsReportPermission]
    renderer_classes = [
        CSVRenderer,
        XLSRenderer,
    ] + api_settings.DEFAULT_RENDERER_CLASSES

    def __init__(self, *args, **kwargs):
        self.filter_params = {}
        super().__init__(*args, **kwargs)

    def get_renderer_context(self):
        context = super().get_renderer_context()
        fields = self.filter_params.get('_fields')
        if fields:
            fields = fields.split(',')
        if not fields:
            fields = self.list_item_serializer_class.Meta.fields
        context['header'] = fields
        return context

    def get_queryset(self):
        if self.request.user.is_analyst:
            return Interview.unsafe.all()
        return Interview.objects.all()

    def filter_queryset(self, queryset, *args, **kwargs):
        filter_form = serializers.InterviewsReportFilterForm(
            data=self.request.query_params,
        )
        if not filter_form.is_valid():
            raise ResponseError(
                data=filter_form.errors,
                status=status.HTTP_400_BAD_REQUEST,
            )
        self.filter_params = filter_form.validated_data
        interviewers = self.filter_params.get('interviewers')
        recruiters = self.filter_params.get('recruiters')
        states = self.filter_params.get('states')
        date_from = self.filter_params.get('date_from')
        date_to = self.filter_params.get('date_to')
        department_urls = self.filter_params.get('departments')

        states = states.split(',') if states else ['finished']
        queryset = queryset.filter(state__in=states)

        if department_urls:
            department_urls = department_urls.split(',')
            department_ids = list(
                Department.objects
                .filter(url__in=department_urls)
                .values_list('id', flat=True)
            )
            departments = Department.objects.in_trees(department_ids)
            queryset = queryset.filter(interviewer__department__in=departments)

        if interviewers:
            interviewers = interviewers.split(',')
            queryset = queryset.filter(interviewer__username__in=interviewers)

        if recruiters:
            recruiters = recruiters.split(',')
            queryset = queryset.filter(created_by__username__in=recruiters)

        if date_from:
            queryset = queryset.filter(
                Q(finished__gte=date_from) | Q(finished__isnull=True)
            )

        if date_to:
            queryset = queryset.filter(finished__lte=date_to)

        queryset = queryset.exclude(
            consideration__candidate__login=self.request.user.username,
        )
        queryset = queryset.values(
            'id',
            'type',
            'section',
            'created',
            'finished',
            'event_start_time',
            'state',
            'grade',
            'aa_type',
            'is_code',
            'is_pro_level_scale',
            'consideration_id',
            'consideration__candidate_id',
            'consideration__candidate__first_name',
            'consideration__candidate__last_name',
            'consideration__candidate__login',
            'application__vacancy_id',
            'application__vacancy__name',
            'application__vacancy__department__url',
            'application__vacancy__profession__name',
            'application__vacancy__pro_level_max',
            'interviewer__username',
            'created_by__username',
        )
        return queryset.order_by('id')

    def get_list_item_serializer_context(self):
        vacancy_cities_map = defaultdict(list)
        for vac_id, city in VacancyCity.objects.values_list('vacancy_id', 'city__name_ru'):
            vacancy_cities_map[vac_id].append(city)
        return {
            'vacancy_cities_map': vacancy_cities_map,
        }

    def get(self, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())
        data = self.get_list_item_serializer_object(queryset).data
        return Response(data)

    def finalize_response(self, request, response, *args, **kwargs):
        response = super().finalize_response(
            request, response, *args, **kwargs)
        if response.status_code >= 400:
            response.accepted_renderer = UTF8JSONRenderer()
        format = response.accepted_renderer.format
        if format in ('csv', 'xls'):
            response['Content-Disposition'] = (
                'attachment; filename=report.{}'.format(format)
            )
        return response


class InterviewProblemsContextMixin:

    def get_problems_context(self):
        interview = self.get_object()
        problems_usage_in_context = get_problems_usage_in_context(interview=interview)
        problem_ids = [a.problem_id for a in interview.assignments.all()]
        return {
            'problems_usage_in_context': problems_usage_in_context,
            'active_complaints_counts': get_active_complaints_counts(problem_ids),
            'problems_stats': get_problems_stats(problem_ids),
        }


class InterviewDetailView(InterviewProblemsContextMixin, InstanceFormViewMixin, BaseView):

    model_class = Interview
    detail_serializer_class = serializers.InterviewDetailSerializer
    form_serializer_class = serializers.InterviewUpdateFormSerializer
    permission_classes = [IsAuthenticated, InterviewPermission]
    validator_classes_map = {
        INTERVIEW_TYPES.hr_screening: forms.HRScreeningUpdateForm,
        INTERVIEW_TYPES.aa: forms.AAInterviewUpdateForm,
    }

    def get_validator_class(self):
        # TODO: удалить после перехода на InterviewCommentForm
        if self.request.method == 'PATCH':
            return forms.InterviewCommonUpdateForm
        return self.validator_classes_map.get(
            self.get_object().type,
            forms.InterviewUpdateForm,
        )

    def get_validator_context(self):
        interview = self.get_object()
        return {
            'interview': interview,
            'candidate': interview.candidate,
            'user_ticket': self.user_ticket,
        }

    def get_detail_serializer_context(self):
        candidate_id = self.get_object().candidate_id
        context = self.get_problems_context()
        context['extended_statuses'] = get_candidates_extended_statuses([candidate_id])
        return context

    def get(self, request, *args, **kwargs):
        """
        Одно собеседование
        """
        return self.retrieve(request, *args, **kwargs)

    @action_logged('interview_update')
    def patch(self, request, *args, **kwargs):
        """
        Частичное редактирование собеседования
        """
        return self.partial_update(request, *args, **kwargs)

    @action_logged('interview_update')
    def put(self, request, *args, **kwargs):
        """
        Редактирование собеседования
        """
        return self.update(request, *args, **kwargs)

    def perform_update(self, data, instance):
        workflow = InterviewWorkflow(instance, self.request.user)
        update_action = workflow.get_action('update')
        if not update_action.is_available():
            raise PermissionDenied('workflow_error')

        return update_action.perform(**data)


class InterviewEventView(BaseView):

    def get(self, request, *args, **kwargs):
        """
        Календарная встреча секции
        """
        event_id = kwargs.get('event_id')
        if event_id is None:
            interview = get_object_or_40x(Interview, pk=kwargs.get('pk'))
            event_id = interview.event_id

        if event_id:
            instance_start_ts = request.query_params.get('instanceStartTs')
            event = get_event(
                event_id=event_id,
                uid=request.user.uid,
                user_ticket=self.user_ticket,
                instance_start_ts=instance_start_ts,
            )
            if event is not None:
                return Response(event.to_dict())
        return Response({})


class InterviewCreateFormView(CandidateHelperFormMixin, BaseFormView):

    validator_class = forms.InterviewCreateForm
    permission_classes = [
        NoDuplicatesPermission,
        ForbiddenForAutohire('interview_create'),
    ]

    def get_validator_context(self):
        return {
            'candidate': self.candidate,
        }


class InterviewCommentListCreateView(CommentListCreateBaseView):

    related_model_class = Interview
    related_workflow_class = InterviewWorkflow
    action_name = 'comment_add'
    actionlog_name = 'interview_comment_add'


class InterviewWorkflowView(InterviewProblemsContextMixin, WorkflowView):

    model_class = Interview
    detail_serializer_class = serializers.InterviewDetailSerializer
    workflow_class = InterviewWorkflow
    permission_classes = [IsAuthenticated, InterviewPermission]

    @property
    def actionlog_name(self):
        return 'interview_%s' % self.action_name

    def get_detail_serializer_context(self):
        candidate_id = self.get_object().candidate_id
        context = self.get_problems_context()
        context['extended_statuses'] = get_candidates_extended_statuses([candidate_id])
        return context

    def get_validator_context(self):
        return {
            'interview': self.get_object(),
        }


class InterviewFinishView(InterviewWorkflowView):

    validator_class = forms.InterviewFinishForm
    action_name = 'finish'

    def get_queryset(self):
        return (
            super().get_queryset()
            .select_related('application__vacancy')
        )


class InterviewCancelView(InterviewWorkflowView):

    validator_class = forms.InterviewCancelForm
    action_name = 'cancel'


class InterviewEstimateView(InterviewWorkflowView):

    validator_class = forms.InterviewEstimateForm
    action_name = 'estimate'


class InterviewCommentView(InterviewWorkflowView):

    validator_class = forms.InterviewCommentForm
    action_name = 'comment'


class InterviewReassignView(InstanceFormViewMixin, InterviewWorkflowView):

    action_name = 'reassign'
    form_serializer_class = serializers.InterviewReassignFormSerializer

    def get_validator_class(self):
        instance = self.get_object()
        return forms.AAInterviewerReassignForm if instance.is_aa else forms.InterviewReassignForm

    def get_validator_context(self):
        return {
            'interview': self.get_object(),
        }


class InterviewReorderView(InterviewProblemsContextMixin, BaseView):

    model_class = Interview
    permission_classes = [IsAuthenticated, InterviewPermission]
    detail_serializer_class = serializers.InterviewDetailSerializer

    def get_detail_serializer_context(self):
        return self.get_problems_context()

    @action_logged('interview_reorder')
    def post(self, request, *args, **kwargs):
        """
        Изменение порядка задач в собеседовании
        """
        interview = self.get_object()
        assignment_ids = self.request.data.get('assignment_ids')

        reorder_interview(interview, assignment_ids)

        serializer = self.get_detail_serializer_object(instance=interview)
        return Response(
            data=serializer.data,
            status=status.HTTP_200_OK,
        )


class InterviewChangeResolutionView(InstanceFormViewMixin, InterviewWorkflowView):

    action_name = 'change_resolution'
    validator_class = forms.InterviewChangeResolutionForm
    form_serializer_class = serializers.InterviewChangeResolutionFormSerializer


class InterviewRenameView(InstanceFormViewMixin, InterviewWorkflowView):

    action_name = 'rename'
    validator_class = forms.InterviewRenameForm
    form_serializer_class = serializers.InterviewRenameFormSerializer


class InterviewAddProblemsFromPreset(InterviewWorkflowView):

    action_name = 'add_problems_from_preset'
    validator_class = AddProblemsFromPresetForm


class InterviewSendToReviewView(BaseFormView, InterviewWorkflowView):

    action_name = 'send_to_review'
    validator_class = forms.InterviewSendToReviewForm
    permission_classes = [IsAuthenticated]


class InterviewChangeGradeView(InstanceFormViewMixin, InterviewWorkflowView):

    action_name = 'change_grade'
    validator_class = forms.InterviewChangeGradeForm
    form_serializer_class = serializers.InterviewChangeGradeFormSerializer


class InterviewChooseScaleView(InstanceFormViewMixin, InterviewWorkflowView):

    action_name = 'choose_scale'
    validator_class = forms.InterviewChooseScaleForm
    form_serializer_class = serializers.InterviewChooseScaleFormSerializer
