import waffle

from collections import defaultdict
from itertools import groupby
from operator import itemgetter

from django.utils.functional import cached_property

from intranet.femida.src.applications.models import Application
from intranet.femida.src.core.switches import TemporarySwitch
from intranet.femida.src.interviews.choices import INTERVIEW_TYPES
from intranet.femida.src.interviews.models import Interview
from intranet.femida.src.vacancies.models import VacancyMembership
from intranet.femida.src.vacancies.choices import VACANCY_STATUSES
from intranet.femida.src.candidates.models import (
    CandidateAttachment,
    CandidateContact,
    CandidateJob,
    CandidateEducation,
    CandidateProfession,
    CandidateResponsible,
    CandidateSkill,
    CandidateTag,
    CandidateCity,
    Consideration,
)


class NestedFetcher:

    model_class = None
    values_fields = ()

    def __init__(self, cand_fetcher):
        self.cand_fetcher = cand_fetcher

    def filter_collection(self, collection):
        return collection

    @cached_property
    def collection(self):
        if hasattr(self.model_class, 'unsafe'):
            manager = self.model_class.unsafe
        else:
            manager = self.model_class.objects
        result = (
            manager
            .filter(
                candidate_id__gte=self.cand_fetcher.min_cand_id,
                candidate_id__lte=self.cand_fetcher.max_cand_id,
            )
            .values(*self.values_fields)
            .order_by('-candidate_id', 'id')
        )
        if self.cand_fetcher.extra_query is not None:
            result = result.filter(self.cand_fetcher.extra_query)
        return self.filter_collection(result)

    def get_repacked_data(self, objs):
        return {}


class AttachmentsFetcher(NestedFetcher):

    model_class = CandidateAttachment
    values_fields = (
        'id',
        'candidate_id',
        'attachment__id',
        'attachment__text',
    )

    def get_repacked_data(self, objs):
        return {
            'attachments': [
                {
                    'id': obj['attachment__id'],
                    'text': obj['attachment__text'],
                }
                for obj in objs
            ],
        }


class ContactsFetcher(NestedFetcher):

    model_class = CandidateContact
    values_fields = (
        'id',
        'candidate_id',
        'type',
        'account_id',
        'is_main',
        # TODO: можно этот флаг удалить (FEMIDA-2731)
        'is_active',
    )

    def filter_collection(self, collection):
        return collection.filter(is_active=True)

    def get_repacked_data(self, objs):
        return {
            'contacts': objs,
        }


class JobsFetcher(NestedFetcher):

    model_class = CandidateJob
    values_fields = (
        'id',
        'candidate_id',
        'employer',
        'start_date',
        'salary_evaluation',
        'end_date',
        'position',
    )

    def get_repacked_data(self, objs):
        return {
            'jobs': objs,
        }


class EducationsFetcher(NestedFetcher):

    model_class = CandidateEducation
    values_fields = (
        'id',
        'candidate_id',
        'faculty',
        'degree',
        'end_date',
        'institution',
    )

    def get_repacked_data(self, objs):
        return {
            'educations': objs,
        }


class SkillsFetcher(NestedFetcher):

    model_class = CandidateSkill
    values_fields = (
        'id',
        'candidate_id',
        'skill__id',
        'skill__name',
    )

    def get_repacked_data(self, objs):
        return {
            'skills': [
                {
                    'id': obj['skill__id'],
                    'name': obj['skill__name'],
                }
                for obj in objs
            ]
        }


class TargetCitiesFetcher(NestedFetcher):

    model_class = CandidateCity
    values_fields = (
        'candidate_id',
        'city_id',
        'city__name_ru',
        'city__name_en',
    )

    def get_repacked_data(self, objs):
        return {
            'target_cities': [
                {
                    'id': obj['city_id'],
                    'name_ru': obj['city__name_ru'],
                    'name_en': obj['city__name_en'],
                }
                for obj in objs
            ]
        }


class TagsFetcher(NestedFetcher):

    model_class = CandidateTag
    values_fields = (
        'candidate_id',
        'tag_id',
        'tag__name',
    )

    def filter_collection(self, collection):
        return collection.filter(is_active=True)

    def get_repacked_data(self, objs):
        # TODO: keep only first branch after FEMIDA-7204 release and isearch change
        if waffle.switch_is_active(TemporarySwitch.ENABLE_NEW_ISEARCH_TAGS_SLUGS):
            return {
                'tags': [
                    {'slug': str(tag['tag_id']), 'name': tag['tag__name']}
                    for tag in objs
                ],
            }
        else:
            return {
                'tags': [tag['tag__name'] for tag in objs],
            }


class ProfessionsAndSpheresFetcher(NestedFetcher):

    model_class = CandidateProfession
    values_fields = (
        'id',
        'candidate_id',
        'profession__id',
        'profession__name',
        'profession__name_en',
        'profession__professional_sphere__id',
        'profession__professional_sphere__name',
        'profession__professional_sphere__name_en',
    )

    def get_repacked_data(self, objs):
        sphere_ids = set()
        data = {
            'professions': [],
            'professional_spheres': [],
        }
        for obj in objs:
            sphere_id = obj['profession__professional_sphere__id']
            data['professions'].append({
                'id': obj['profession__id'],
                'name': obj['profession__name'],
                'name_en': obj['profession__name_en'],
                'professional_sphere_id': sphere_id,
            })
            if sphere_id not in sphere_ids:
                sphere_ids.add(sphere_id)
                data['professional_spheres'].append({
                    'id': sphere_id,
                    'name': obj['profession__professional_sphere__name'],
                    'name_en': obj['profession__professional_sphere__name_en'],
                })
        return data


class ResponsiblesFetcher(NestedFetcher):

    model_class = CandidateResponsible
    values_fields = (
        'id',
        'candidate_id',
        'user__username',
    )

    def get_repacked_data(self, objs):
        return {
            'responsibles': [obj['user__username'] for obj in objs]
        }


class ApplicationsAndVacanciesFetcher(NestedFetcher):

    model_class = Application
    values_fields = (
        'id',
        'candidate_id',
        'status',
        'resolution',
        'vacancy__id',
        'vacancy__name',
        'vacancy__status',
        'vacancy__is_hidden',
    )

    @cached_property
    def vacancy_access_map(self):
        memberships = (
            VacancyMembership.unsafe
            .filter(vacancy_id__in=self.collection.values('vacancy__id'))
            .values('vacancy_id', 'member__username')
        )
        result = defaultdict(list)
        for item in memberships:
            result[item['vacancy_id']].append(item['member__username'])
        return result

    def get_repacked_data(self, objs):
        data = {
            'applications': [],
            'vacancies': [],
        }
        has_active_vacancies = False
        has_active_visible_vacancies = False
        vacancy_ids = []

        for obj in objs:
            data['applications'].append({
                'id': obj['id'],
                'status': obj['status'],
                'is_active': obj['status'] != VACANCY_STATUSES.closed,
                'is_archived': obj['status'] == VACANCY_STATUSES.closed,
                'resolution': obj['resolution'],
                'vacancy_id': obj['vacancy__id'],
            })
            vacancy = {
                'id': obj['vacancy__id'],
                'name': obj['vacancy__name'],
                'status': obj['vacancy__status'],
                'is_hidden': obj['vacancy__is_hidden'],
            }
            data['vacancies'].append(vacancy)
            if vacancy['status'] != 'closed':
                has_active_vacancies = True
                if not vacancy['is_hidden']:
                    has_active_visible_vacancies = True

            vacancy_ids.append(vacancy['id'])

        data['has_visible_vacancies'] = has_active_visible_vacancies
        data['has_only_hidden_vacancies'] = (
            has_active_vacancies and not has_active_visible_vacancies
        )

        for vacancy in data['vacancies']:
            vacancy['access'] = self.vacancy_access_map.get(vacancy['id'], [])
        return data


class InterviewsFetcher(NestedFetcher):

    model_class = Interview
    values_fields = (
        'id',
        'type',
        'candidate_id',
        'consideration_id',
        'consideration__state',
        'state',
        'grade',
        'interviewer__username',
    )

    def filter_collection(self, collection):
        # Здесь фильтруем только по типу и статусу самих секций
        # Фильтр по статусу рассмотрения в get_repacked_data в коде, потому что иначе
        # запросы будут медленными
        return collection.gradable().filter(state=Interview.STATES.finished)

    def get_repacked_data(self, objs):
        data = {
            'interviews': [],
            'hire_interviews_count': 0,
            'nohire_interviews_count': 0,
            'skype_interviews_count': 0,
            'on_site_interviews_count': 0,
            'hire_interviews_avg_grade': None,
            'skype_interviews_avg_grade': None,
            'on_site_interviews_avg_grade': None,
        }
        all_grades = []
        skype_grades = []
        on_site_grades = []

        # Если в активном рассмотрении есть секции в статусе assigned или estimated,
        # то не добавляем в список extra_ids секции этого рассмотрения (кроме screening)
        # Секции из этого списка добавлять в выдачу не будем
        extra_ids = set()
        is_consideration_completed = True
        for obj in objs:
            if obj['consideration__state'] == Consideration.STATES.archived:
                continue
            if obj['type'] == INTERVIEW_TYPES.screening:
                continue
            if obj['state'] in [Interview.STATES.assigned, Interview.STATES.estimated]:
                is_consideration_completed = False
            extra_ids.add(obj['id'])

        if is_consideration_completed:
            extra_ids = set()

        for obj in objs:
            if obj['id'] in extra_ids:
                continue

            interview = {
                'id': obj['id'],
                'state': obj['state'],
                'grade': obj['grade'],
                'interviewer': obj['interviewer__username'],
            }
            data['interviews'].append(interview)

            if obj['type'] == 'screening':
                data['skype_interviews_count'] += 1
                if obj['grade']:
                    skype_grades.append(obj['grade'])
            else:
                data['on_site_interviews_count'] += 1
                if obj['grade']:
                    on_site_grades.append(obj['grade'])

            if obj['grade']:
                data['hire_interviews_count'] += 1
                all_grades.append(obj['grade'])
            else:
                data['nohire_interviews_count'] += 1

        if all_grades:
            data['hire_interviews_avg_grade'] = sum(all_grades) / float(len(all_grades))
        if skype_grades:
            data['skype_interviews_avg_grade'] = sum(skype_grades) / float(len(skype_grades))
        if on_site_grades:
            data['on_site_interviews_avg_grade'] = sum(on_site_grades) / float(len(on_site_grades))
        return data


class CandidatesFetcher:

    nested_fetcher_classes = (
        AttachmentsFetcher,
        ContactsFetcher,
        JobsFetcher,
        EducationsFetcher,
        ResponsiblesFetcher,
        ProfessionsAndSpheresFetcher,
        SkillsFetcher,
        ApplicationsAndVacanciesFetcher,
        InterviewsFetcher,
        TargetCitiesFetcher,
        TagsFetcher,
    )

    def __init__(self, base_data, extra_query=None):
        self._data = base_data
        self.min_cand_id = 2 ** 31
        self.max_cand_id = 0
        self.data_dict = {}
        self.extra_query = extra_query

        for item in self._data:
            _id = item['id']
            if _id < self.min_cand_id:
                self.min_cand_id = _id
            if _id > self.max_cand_id:
                self.max_cand_id = _id
            self.data_dict[_id] = item

    @cached_property
    def data(self):
        for fetcher_class in self.nested_fetcher_classes:
            fetcher = fetcher_class(cand_fetcher=self)
            self.process_nested_fetcher(fetcher)
        return self._data

    def process_nested_fetcher(self, fetcher):
        groups = groupby(fetcher.collection, itemgetter('candidate_id'))
        collection_map = {cand_id: list(items) for cand_id, items in groups}
        for candidate_id, data in self.data_dict.items():
            items = collection_map.get(candidate_id, [])
            data.update(fetcher.get_repacked_data(items))
