from collections import defaultdict
from datetime import datetime

import pytz
import waffle

from constance import config
from django.conf import settings
from django.db.models import F, Q, Exists, OuterRef, Value, BooleanField
from django.db.models.base import ModelBase
from django.http import Http404
from django.utils.decorators import method_decorator
from django_replicated.decorators import use_master
from rest_framework import status
from rest_framework.response import Response

from intranet.femida.src.actionlog.manager import mongo
from intranet.femida.src.api.core.errors import format_non_field_message
from intranet.femida.src.api.core.views import BaseView
from intranet.femida.src.candidates.models import CandidateProfession, Consideration
from intranet.femida.src.candidates.choices import CONSIDERATION_STATUSES, CANDIDATE_STATUSES
from intranet.femida.src.celery_app import app
from intranet.femida.src.interviews.models import Interview
from intranet.femida.src.monitoring.checks import celery as celery_checks
from intranet.femida.src.offers.choices import OFFER_STATUSES, EXTERNAL_EMPLOYEE_TYPES
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.permissions.helpers import get_manager
from intranet.femida.src.staff.choices import DEPARTMENT_ROLES
from intranet.femida.src.staff.models import DepartmentUser
from intranet.femida.src.vacancies.models import Vacancy


class MonitoringBaseView(BaseView):

    authentication_classes = []
    permission_classes = []


@method_decorator(use_master, 'get')
class MonitoringCeleryWorkerView(MonitoringBaseView):
    """
    Проверяет состояние хоста, на котором запущен celery worker, через заданные проверки.
    Если все проверки успешны – позволяем воркеру получать новые таски.
    Если нет – запрещаем.
    """
    checks = [
        celery_checks.check_db_read,
    ]

    def get(self, request, *args, **kwargs):
        # FIXME: Пока что не включаем эту проверку. Доделываем в FEMIDA-5670
        if not waffle.switch_is_active('enable_celery_worker_monitoring'):
            raise Http404
        check_results = {c.__name__: c() for c in self.checks}
        if all(check_results.values()):
            app.start_consuming()
            return Response(check_results)
        else:
            app.stop_consuming()
            return Response(check_results, status.HTTP_400_BAD_REQUEST)


class MonitoringDBConsistencyView(MonitoringBaseView):

    consistency_check_data = (
        (Interview, 'candidate_id', 'application__candidate_id'),
        (CandidateProfession, 'professional_sphere_id', 'profession__professional_sphere_id'),
        (Offer, 'candidate_id', 'application__candidate_id'),
        (Offer, 'vacancy_id', 'application__vacancy_id'),
        (
            Vacancy.unsafe.filter(profession__isnull=False),
            'professional_sphere_id',
            'profession__professional_sphere_id',
        ),
    )

    def get(self, request, *args, **kwargs):
        """
        Ручка мониторинга неконсистентности денормализованных данных
        """
        inconsistent_data = []
        for model_or_qs, field1, field2 in self.consistency_check_data:
            qs = (
                get_manager(model_or_qs, unsafe=True)
                if isinstance(model_or_qs, ModelBase)
                else model_or_qs
            )
            if qs.exclude(**{field1: F(field2)}).exists():
                inconsistent_data.append({
                    'model': qs.model.__name__,
                    'fields': [field1, field2],
                })

        if inconsistent_data:
            data = dict(
                format_non_field_message('inconsistent_data'),
                inconsistent_data=inconsistent_data,
            )
            return Response(data, status.HTTP_400_BAD_REQUEST)

        return Response({})


class MonitoringOfferConsistencyView(MonitoringBaseView):

    def get(self, request, *args, **kwargs):
        """
        Ручка мониторинга неконсистентности данных в офферах
        """
        inconsistent_offers = []
        inconsistent_data = defaultdict(list)

        active_offers_qs = (
            Offer.unsafe
            .alive()
            .exclude(status=OFFER_STATUSES.draft)
            .exclude(
                # Нам неинтересно, что с данными офферов,
                # которые уже отправлены в Наниматор.
                # После того, как это случилось, удалённые
                # офисы, подразделения и т.д. нам уже никак не помешают
                status=OFFER_STATUSES.accepted,
                newhire_id__isnull=False,
            )
        )

        # Офферы, которые ссылаются на
        # удаленную организацию, удаленный офис,
        # или удаленное подразделение
        inconsistent_offers.extend(
            active_offers_qs
            .filter(
                Q(org__is_deleted=True)
                | Q(office__is_deleted=True)
                | Q(department__is_deleted=True)
            )
            .values(
                'id',
                has_deleted_org=F('org__is_deleted'),
                has_deleted_office=F('office__is_deleted'),
                has_deleted_department=F('department__is_deleted'),
            )
        )

        # Офферы, у которых нет руководителя
        inconsistent_offers.extend(
            active_offers_qs
            .annotate(
                has_no_boss=~Exists(
                    DepartmentUser.objects
                    .filter(
                        department=OuterRef('department'),
                        role=DEPARTMENT_ROLES.chief,
                    )
                    .values('id')
                ),
            )
            .filter(has_no_boss=True)
            .values('id', 'has_no_boss')
        )

        # Принятые и закрытые офферы без HR-тикета
        inconsistent_offers.extend(
            Offer.unsafe
            .filter(
                # Начинаем отслеживать неконсистентность с текущей даты, так как в
                # исторических данных могут быть офферы без HR-тикетов
                created__gte=datetime(2019, 12, 25, tzinfo=pytz.utc),
                startrek_hr_key='',
                status__in=(
                    OFFER_STATUSES.accepted,
                    OFFER_STATUSES.closed,
                ),
                employee_type__in=EXTERNAL_EMPLOYEE_TYPES._db_values,
            )
            .values('id', has_no_hr_issue=Value(True, output_field=BooleanField()))
        )

        for offer in inconsistent_offers:
            offer_id = offer.pop('id')
            for k, v in offer.items():
                if v:
                    inconsistent_data[k].append(offer_id)

        if inconsistent_data:
            data = dict(
                format_non_field_message('inconsistent_data'),
                inconsistent_data=inconsistent_data,
            )
            return Response(data, status.HTTP_400_BAD_REQUEST)

        return Response({})


class MonitoringCandidateConsistencyView(MonitoringBaseView):

    def get(self, request, *args, **kwargs):
        """
        Ручка мониторинга неконсистентности данных в кандидатах/рассмотрениях
        """
        inconsistent_data = []

        # Закрытые кандидаты, у которых есть открытые рассмотрения
        inconsistent_data.extend(
            Consideration.unsafe
            .filter(
                state=CONSIDERATION_STATUSES.in_progress,
                candidate__status=CANDIDATE_STATUSES.closed,
            )
            .values('candidate_id')
        )

        if inconsistent_data:
            data = dict(
                format_non_field_message('inconsistent_data'),
                inconsistent_data=inconsistent_data,
            )
            return Response(data, status.HTTP_400_BAD_REQUEST)

        return Response({})


class MonitoringCeleryTaskQueueSizeView(MonitoringBaseView):

    def get(self, request, *args, **kwargs):
        collection = mongo.db.messages
        queue_size = collection.count_documents({'queue': settings.CELERY_TASK_DEFAULT_QUEUE})
        pending_messages_count = collection.count_documents(
            filter={'queue': settings.CELERY_TASK_DEFAULT_QUEUE, 'hidden_till': None},
        )

        if queue_size > config.CELERY_QUEUE_SIZE_THRESHOLD:
            data = {
                'queue_size': queue_size,
                'pending_messages_count': pending_messages_count,
            }
            return Response(data, status=status.HTTP_400_BAD_REQUEST)

        return Response({})
