"""
Общие базовые формы для офферов и препрофайлов.
"""
import sform

from django.conf import settings
from django.conf.locale.en.formats import DATE_INPUT_FORMATS as EN_DATE_INPUT_FORMATS
from django.conf.locale.ru.formats import DATE_INPUT_FORMATS as RU_DATE_INPUT_FORMATS

from intranet.femida.src.api.core.forms import NumericCharField
from intranet.femida.src.api.core.validators import validate_phone
from intranet.femida.src.offers.validators import name_en_validator


# Выходит костыльно. Лучше, наверное, указывать верный формат на фронте
OFFER_DATE_INPUT_FORMATS = EN_DATE_INPUT_FORMATS + RU_DATE_INPUT_FORMATS


class UsernameField(sform.CharField):

    min_length = settings.MIN_LOGIN_LENGTH
    max_length = settings.MAX_LOGIN_LENGTH

    def __init__(self, *args, **kwargs):
        kwargs.setdefault('min_length', self.min_length)
        kwargs.setdefault('max_length', self.max_length)
        super().__init__(*args, **kwargs)

    def structure_as_dict(self, *args, **kwargs):
        result = super().structure_as_dict(*args, **kwargs)
        result['type'] = 'char'
        result['minlength'] = self.kwargs['min_length']
        return result


class InternalFormUsernameField(UsernameField):
    """
    Поле для логина, которое заполняется рекрутерами на внутренних формах.
    Для бывших и действующих сотрудников длина логина может
    не вписываться в действующие рамки – логин может быть меньше 4 и больше 13 символов.
    Поэтому ставим реально допустимые рамки без привязки к текущим правилам.
    """
    min_length = 0
    max_length = 30


class NameValidationFormMixin:

    def _clean_name(self, value):
        # Удаляем двойные пробелы из имени, на которые ругается Наниматор
        return ' '.join(value.split())

    def _clean_name_en(self, value):
        # TODO: починить на уровне sform.
        # Если вынести это в валидаторы поля
        # то все ломается, из-за deepcopy(base_fields) в sform,
        # т.к. в данном валидаторе есть скомпилированный regex,
        # который не копируется.
        # В drf была аналогичная проблема
        # В качестве примера можно использовать https://goo.gl/UdYyDn.
        name_en_validator(value)
        return self._clean_name(value)

    clean_last_name = _clean_name
    clean_first_name = _clean_name
    clean_last_name_en = _clean_name_en
    clean_first_name_en = _clean_name_en


class BankDetailsFormMixin(sform.SForm):
    """
    Миксин формы с банковскими реквизитами
    """
    BANK_FIELD_NAMES = (
        'bic',
        'bank_name',
        'bank_account',
    )

    bic = NumericCharField(min_length=9, max_length=9)
    bank_name = sform.CharField(max_length=80)
    bank_account = NumericCharField(min_length=20, max_length=20)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        if not self.is_bank_details_needed():
            for field_name in self.BANK_FIELD_NAMES:
                self.fields.pop(field_name)

    def is_bank_details_needed(self):
        """
        Требуется переопределение в подклассах:
        возвращает True, если банковские реквизиты нужны.
        """
        raise NotImplementedError

    def clean(self):
        # Собираем все это в отдельный словарь прямо здесь,
        # так с этим легче работать дальше
        cd = super().clean()
        cd['bank_details'] = {}
        for field_name in self.BANK_FIELD_NAMES:
            cd['bank_details'][field_name] = cd.pop(field_name, '')
        return cd


class EdsPhoneForm(sform.SForm):
    eds_phone = sform.CharField(
        max_length=50,
        state=sform.REQUIRED,
        validators=[validate_phone],
    )


class EdsCodeForm(sform.SForm):
    code = sform.CharField(state=sform.REQUIRED)
