"""
Общие базовые вьюхи для офферов и препрофайлов.
Возможно, стоит их вынести из api.
"""
import os

from constance import config
from django.conf import settings
from rest_framework import status
from rest_framework.parsers import MultiPartParser, FormParser
from rest_framework.response import Response

from intranet.femida.src.actionlog.models import actionlog
from intranet.femida.src.api.core import errors
from intranet.femida.src.api.core.serializers import IdNameSerializer
from intranet.femida.src.api.core.views import BaseView, ResponseError
from intranet.femida.src.offers.choices import OFFER_CANDIDATE_ATTACHMENT_TYPES
from intranet.femida.src.offers.mixins import throttle_per_path
from intranet.femida.src.core.controllers import update_instance
from intranet.femida.src.utils.newhire import NewhireAPI

from .base_forms import EdsCodeForm, EdsPhoneForm

if os.getenv('IS_ARCADIA'):
    from django_mds.client import APIError
else:
    from mds import APIError


class AttachmentUploadBaseView(BaseView):

    parser_classes = (MultiPartParser, FormParser,)
    authentication_classes = []
    permission_classes = []
    actionlog_name = None
    controller_class = None

    def _raise_400(self, message, field='file'):
        raise ResponseError(
            data=errors.format_message(field, message),
            status=status.HTTP_400_BAD_REQUEST,
        )

    def perform_upload(self, request, *args, **kwargs):
        if 'file' not in request.FILES:
            self._raise_400(errors.NO_FILES_GIVEN_IN_REQUEST)

        attachment_type = request.data.get('type', 'document')
        if attachment_type not in OFFER_CANDIDATE_ATTACHMENT_TYPES:
            self._raise_400('invalid_choice', 'type')

        file = request.FILES['file']
        if file.size > settings.OFFER_MAX_ATTACHMENT_SIZE * 1024 ** 2:
            self._raise_400('file_too_large')

        try:
            ctl = self.controller_class(self.get_object())
            attached_file = ctl.attach_file(file, attachment_type, is_active=False)
        except APIError:
            self._raise_400(errors.UPLOAD_TO_MDS_FAILED)
        else:
            return attached_file.attachment

    def post(self, request, *args, **kwargs):
        """
        Заливка аттача оффера/препрофайла
        """
        with actionlog.init(self.actionlog_name):
            attachment = self.perform_upload(request, *args, **kwargs)
            data = IdNameSerializer(attachment).data
            return Response(data, status=status.HTTP_200_OK)


class AttachEdsPhoneBaseView(BaseView):

    validator_class = EdsPhoneForm
    actionlog_name = None
    throttle_classes = []

    def post(self, request, *args, **kwargs):
        validator = self.get_validator_object(request.data)
        self.validate(validator)

        # Включаем ограничения только на запросы с корректным телефоном
        self.throttle_classes = [throttle_per_path(config.EDS_SMS_RATE)]
        self.check_throttles(request)

        data = {
            'object_type': self.model_class._meta.model_name,
            'object_id': self.instance.id,
            'phone_number': validator.cleaned_data['eds_phone'],
        }

        response_data, status_code = NewhireAPI.attach_phone(data)
        if status_code != status.HTTP_200_OK:
            error_code = self._get_error_code(response_data)
            return Response(
                data={'error': [{'code': error_code}]},
                status=status.HTTP_400_BAD_REQUEST,
            )

        with actionlog.init(self.actionlog_name):
            update_instance(self.instance, validator.cleaned_data)
            return Response()

    def _get_error_code(self, response_data):
        response_data = response_data or {}
        response_errors = response_data.get('errors', [])
        default_error = 'sms_sending_error'
        if isinstance(response_errors, dict):
            response_errors = response_errors.get('phone_number', [])

        if not response_errors:
            return default_error

        known_errors = {
            'badphone': 'invalid_phone_number',
            'invalid_phone_number': 'invalid_phone_number',
        }
        message = response_errors[0].get('code') or response_errors[0].get('message')
        return known_errors.get(message, default_error)


class VerifyEdsPhoneBaseView(BaseView):

    validator_class = EdsCodeForm
    actionlog_name = None

    def post(self, request, *args, **kwargs):
        validator = self.get_validator_object(request.data)
        self.validate(validator)

        data = {
            'object_type': self.model_class._meta.model_name,
            'object_id': self.instance.id,
            'code': validator.cleaned_data['code'],
        }
        response_data, status_code = NewhireAPI.verify_code(data)
        if status_code != status.HTTP_200_OK:
            error_code = self._get_error_code(response_data)
            return Response(
                data={'error': [{'code': error_code}]},
                status=status.HTTP_400_BAD_REQUEST,
            )

        with actionlog.init(self.actionlog_name):
            update_instance(self.instance, {'is_eds_phone_verified': True})
            return Response()

    def _get_error_code(self, response_data):
        response_data = response_data or {}
        response_errors = response_data.get('errors', [])
        default_error = 'phone_verification_error'
        if isinstance(response_errors, dict):
            response_errors = response_errors.get('code', [])

        if not response_errors:
            return default_error

        message = response_errors[0].get('code') or response_errors[0].get('message')
        return message or default_error
