import logging

from django.utils import translation
from django.utils.functional import cached_property
from rest_framework import status
from rest_framework.exceptions import PermissionDenied
from rest_framework.response import Response

from intranet.femida.src.actionlog.models import actionlog
from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.api.core.views import (
    BaseView,
    InstanceFormViewMixin,
    WorkflowView,
)
from intranet.femida.src.api.core import errors
from intranet.femida.src.api.hire_orders.permissions import ForbiddenForAutohire
from intranet.femida.src.api.offers.base_views import (
    AttachmentUploadBaseView,
    AttachEdsPhoneBaseView,
    VerifyEdsPhoneBaseView,
)
from intranet.femida.src.api.tracker.views import TrackerTriggerBaseView
from intranet.femida.src.candidates.choices import VERIFICATION_STATUSES
from intranet.femida.src.core.choices import DRF_THROTTLE_SCOPES
from intranet.femida.src.core.workflow import WorkflowError
from intranet.femida.src.hire_orders.choices import HIRE_ORDER_STATUSES
from intranet.femida.src.hire_orders.models import HireOrder
from intranet.femida.src.offers.choices import (
    FORM_TYPES,
    OFFER_STATUSES,
    OFFER_DOCS_PROCESSING_STATUSES,
)
from intranet.femida.src.offers.controllers import OfferCtl
from intranet.femida.src.offers.mixins import OfferByUidViewMixin
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.offers.tasks import (
    offer_confirm_by_current_team_task,
    offer_approve_by_issue_task,
    offer_confirm_by_issue_task,
    offer_decline_by_issue_task,
)
from intranet.femida.src.offers.workflow import OfferWorkflow
from intranet.femida.src.staff.bp_registry import BPRegistryError
from intranet.femida.src.vacancies.choices import VACANCY_TYPES
from intranet.femida.src.vacancies.models import Vacancy

from . import forms, serializers


logger = logging.getLogger(__name__)


class BPDataMixin:

    @cached_property
    def bp_data(self):
        instance = self.get_object()
        if instance.vacancy.budget_position_id:
            offer = OfferCtl(instance)
            data = offer.budget_position_data
            if 'payment_currency' in data:
                data['payment_currency'] = data['payment_currency'].code
            return data
        return {}


class NewhireDataMixin:

    @cached_property
    def newhire_data(self):
        return OfferCtl(self.get_object()).newhire_data


class OfferDetailView(BPDataMixin, InstanceFormViewMixin, BaseView):

    model_class = Offer
    detail_serializer_class = serializers.OfferDetailSerializer
    form_serializer_class = serializers.OfferFormSerializer

    def get_validator_context(self):
        offer = self.get_object()
        return {
            'bp_data': self.bp_data,
            'offer': offer,
            'geography_international': offer.vacancy.geography_international,
        }

    def get_validator_class(self):
        instance = self.get_object()
        if instance.status in (OFFER_STATUSES.draft, OFFER_STATUSES.ready_for_approval):
            return forms.OfferForm
        if instance.is_internal:
            return forms.InternalOfferBaseForm
        return forms.ExternalOfferBaseForm

    def get(self, request, *args, **kwargs):
        """
        Детали оффера
        """
        return self.retrieve(request, *args, **kwargs)

    @action_logged('offer_update')
    def put(self, request, *args, **kwargs):
        """
        Редактирование оффера
        """
        try:
            return self.update(request, *args, **kwargs)
        except BPRegistryError:
            data = errors.format_non_field_message('cant_find_schemes')
            return Response(data=data, status=status.HTTP_400_BAD_REQUEST)

    def perform_update(self, data, instance):
        workflow = OfferWorkflow(
            instance=instance,
            user=self.request.user,
        )
        update_action = workflow.get_action('update')
        if not update_action.is_available():
            raise PermissionDenied('workflow_error')

        return update_action.perform(**data)


class OfferListView(BaseView):

    model_class = Offer
    list_item_serializer_class = serializers.OfferListSerializer

    def filter_queryset(self, queryset):
        candidate_id = self.request.query_params.get('candidate_id')
        vacancy_id = self.request.query_params.get('vacancy_id')
        status = self.request.query_params.get('status')
        statuses = status.split(',') if status else []

        vacancies_qs = Vacancy.objects.values('id')
        if vacancy_id:
            vacancies_qs = vacancies_qs.filter(id=vacancy_id)
        queryset = queryset.filter(vacancy__in=vacancies_qs)
        if candidate_id:
            queryset = queryset.filter(candidate_id=candidate_id)
        if statuses:
            queryset = queryset.filter(status__in=statuses)

        return queryset.order_by('-modified')

    def get_queryset(self):
        """
        TODO: Список офферов - исключение в правах.
        Их видимость в списке шире, чем в остальных местах. Поэтому решели пока тут обработать
        отдельно через unsafe с фильтром по objects вакансий.
        """
        return (
            self.model_class.unsafe
            .select_related(
                'creator',
                'application',
                'vacancy',
                'candidate',
            )
        )

    def get(self, request, *args, **kwargs):
        """
        Список офферов
        """
        return self.list(request, *args, **kwargs)


class OfferGeneratedTextView(BaseView):
    """
    Получение сброшенного текста оффера
    """

    model_class = Offer
    detail_serializer_class = serializers.OfferGeneratedTextSerializer

    def get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)


class OfferCheckLoginView(BaseView):

    model_class = Offer
    validator_class = forms.CheckLoginForm

    def get_validator_context(self):
        return {
            'offer': self.get_object(),
        }

    def get(self, request, *args, **kwargs):
        validator = self.get_validator_object(request.query_params)
        self.validate(validator)
        return Response(status=status.HTTP_200_OK)


class OfferCheckLoginExternalView(OfferByUidViewMixin, BaseView):
    """
    Валидациия логина
    """
    authentication_classes = []
    permission_classes = []
    validator_class = forms.CheckLoginExternalForm
    throttle_scope = DRF_THROTTLE_SCOPES.ext_form_check_login

    def get_validator_context(self):
        return {
            'offer': self.offer,
        }

    def get(self, request, *args, **kwargs):
        translation.activate(self.get_language())
        validator = self.get_validator_object(request.query_params)
        self.validate(validator)
        return Response(status=status.HTTP_200_OK)


class OfferAttachmentUploadView(OfferByUidViewMixin, AttachmentUploadBaseView):

    actionlog_name = 'offer_attachment_upload'
    controller_class = OfferCtl
    throttle_scope = DRF_THROTTLE_SCOPES.ext_form_attachment_upload

    def get_object(self):
        return self.offer


# Workflow

class BaseWorkflowFormViewMixin(InstanceFormViewMixin):

    def _prevalidate(self):
        workflow = OfferWorkflow(
            instance=self.get_object(),
            user=self.request.user,
        )
        action = workflow.get_action(self.action_name)
        if not action.is_available():
            raise WorkflowError(action.validator_errors)

    def get_form(self, request, *args, **kwargs):
        try:
            self._prevalidate()
            return super().get_form(request, *args, **kwargs)
        except WorkflowError as exc:
            data = errors.format_non_form_message(exc.message)
            return Response(data, status=status.HTTP_400_BAD_REQUEST)


class OfferWorkflowView(WorkflowView):

    model_class = Offer
    detail_serializer_class = serializers.OfferDetailSerializer
    workflow_class = OfferWorkflow

    @property
    def actionlog_name(self):
        return 'offer_%s' % self.action_name


class OfferApproveView(BaseWorkflowFormViewMixin, BPDataMixin, OfferWorkflowView):

    form_serializer_class = serializers.OfferFormSerializer
    permission_classes = [ForbiddenForAutohire]

    @property
    def action_name(self):
        instance = self.get_object()
        if instance.is_internal:
            return 'approve_by_current_team'
        return 'approve'

    def get_validator_class(self):
        instance = self.get_object()
        if instance.is_internal:
            return forms.InternalOfferApproveForm
        return forms.ExternalOfferApproveForm

    def get_validator_context(self):
        return {
            'bp_data': self.bp_data,
            'offer': self.get_object(),
        }


class OfferReapproveView(OfferWorkflowView):

    action_name = 'reapprove'
    permission_classes = [ForbiddenForAutohire]


class OfferSendView(BaseWorkflowFormViewMixin, OfferWorkflowView):

    action_name = 'send'
    validator_class = forms.OfferSendForm
    form_serializer_class = serializers.OfferSendFormSerializer
    permission_classes = [ForbiddenForAutohire]

    def get_validator_context(self):
        return {
            'offer': self.get_object(),
        }


class OfferAcceptView(BPDataMixin, OfferByUidViewMixin, InstanceFormViewMixin, OfferWorkflowView):

    action_name = 'accept'
    detail_serializer_class = None
    form_serializer_class = serializers.OfferAcceptFormSerializer
    authentication_classes = []
    permission_classes = []
    throttle_scope = DRF_THROTTLE_SCOPES.ext_form_accept

    def initial(self, request, *args, **kwargs):
        translation.activate(self.get_language())
        return super().initial(request, *args, **kwargs)

    def get_object(self):
        return self.offer

    @cached_property
    def is_with_hireorder_autofill(self):
        hire_order = (
            HireOrder.objects
            .exclude(status=HIRE_ORDER_STATUSES.closed)
            .filter(offer=self.get_object())
            .first()
        )
        return hire_order and hire_order.autofill_offer

    @cached_property
    def is_with_autofill(self):
        return self.is_with_hireorder_autofill

    def get_validator_class(self):
        if self.offer.docs_processing_status == OFFER_DOCS_PROCESSING_STATUSES.need_information:
            return forms.OfferShortAcceptForm
        if self.offer.form_type == FORM_TYPES.international:
            return forms.OfferAcceptInternationalForm
        return forms.OfferAcceptRussianForm

    def get_validator_context(self):
        return {
            'offer': self.get_object(),
            'bp_data': self.bp_data,
            'is_with_hireorder_autofill': self.is_with_hireorder_autofill,
        }

    def get_initial_data(self):
        initial_data = super().get_initial_data()
        if not self.is_with_autofill:
            return initial_data
        candidate = self.get_object().candidate
        verification = (
            candidate.verifications
            .alive()
            .exclude(status=VERIFICATION_STATUSES.new)
            .first()
        )
        candidate_data = serializers.CandidateForOfferAcceptFormSerializer(candidate).data
        verification_data = (
            serializers.VerificationForOfferAcceptFormSerializer(verification).data
            if verification
            else {}
        )
        initial_data |= {
            key: verification_data.get(key) or candidate_data.get(key)
            for key in candidate_data.keys() | verification_data
        }
        return initial_data

    def post(self, request, *args, **kwargs):
        with actionlog.init(self.actionlog_name, request.user):
            ctl = OfferCtl(self.get_object(), request.user)
            ctl.update_link_counter()
            return self.perform_action(self.action_name)


class OfferUpdateJoinAtView(InstanceFormViewMixin, NewhireDataMixin, OfferWorkflowView):

    action_name = 'update_join_at'
    validator_class = forms.OfferUpdateJoinAtForm

    def get_initial_data(self):
        return self.newhire_data


class OfferUpdateUsernameView(InstanceFormViewMixin, NewhireDataMixin, OfferWorkflowView):

    action_name = 'update_username'
    validator_class = forms.OfferUpdateUsernameForm

    def get_initial_data(self):
        return self.newhire_data

    def get_validator_context(self):
        return {
            'offer': self.get_object(),
        }


class OfferUpdateDepartmentView(InstanceFormViewMixin, OfferWorkflowView):

    action_name = 'update_department'
    validator_class = forms.OfferUpdateDepartmentForm
    form_serializer_class = serializers.OfferUpdateDepartmentFormSerializer

    def get_validator_context(self):
        return {
            'offer': self.get_object(),
        }


class OfferDeleteView(BaseWorkflowFormViewMixin, OfferWorkflowView):

    action_name = 'delete'
    validator_class = forms.OfferDeleteForm
    permission_classes = [ForbiddenForAutohire]

    def get_initial_data(self):
        return {}


class OfferRejectView(BaseWorkflowFormViewMixin, OfferWorkflowView):

    action_name = 'reject'
    validator_class = forms.OfferRejectForm
    form_serializer_class = serializers.OfferRejectFormSerializer
    permission_classes = [ForbiddenForAutohire]


class OfferConfirmByCurrentTeamView(TrackerTriggerBaseView):
    """
    OK-согласование оффера в SALARY-тикете завершено.
    Отдающая сторона подтверждает ротацию
    """
    model_class = Offer
    issue_key_field = 'startrek_salary_key'
    task = offer_confirm_by_current_team_task
    valid_statuses = (
        OFFER_STATUSES.on_rotation_approval,
    )


class OfferApproveByIssueView(TrackerTriggerBaseView):
    """
    Начальное состояние:
      OK-согласование оффера в JOB-тикете завершено. JOB-тикет находится в статусе `hr_approval`.
    Действие:
      Создаем запись в реестре.
    """
    model_class = Offer
    issue_key_field = 'vacancy__startrek_key'
    task = offer_approve_by_issue_task
    valid_statuses = (
        OFFER_STATUSES.on_approval,
    )


class OfferConfirmByIssueView(TrackerTriggerBaseView):
    """
    Начальное состояние:
      Аналитик подтвердил изменения в реестре. JOB-тикет находится в статусе `resolved`.
    Действие:
      Валидируем данные. При обнаружении ошибок откатываем тикет в статус `hr_approval`.
    """
    model_class = Offer
    issue_key_field = 'vacancy__startrek_key'
    task = offer_confirm_by_issue_task
    valid_statuses = (
        OFFER_STATUSES.on_approval,
    )


class OfferDeclineByIssueView(TrackerTriggerBaseView):
    """
    Оффер не согласован в JOB-тикете.
    Данная ручка сейчас актуальна только для автонайма.
    """
    model_class = Offer
    issue_key_field = 'vacancy__startrek_key'
    valid_statuses = (
        OFFER_STATUSES.on_approval,
    )
    task = offer_decline_by_issue_task
    is_retriable = False

    def get_queryset(self):
        queryset = super().get_queryset()
        return queryset.filter(vacancy__type=VACANCY_TYPES.autohire)


class OfferAttachEdsPhoneView(OfferByUidViewMixin, AttachEdsPhoneBaseView):
    """
    Ручка прикрепления телефона для электронной подписи
    """
    actionlog_name = 'offer_attach_eds_phone'
    authentication_classes = []
    permission_classes = []
    model_class = Offer
    throttle_classes = []

    @property
    def instance(self):
        return self.offer


class OfferVerifyEdsPhoneView(OfferByUidViewMixin, VerifyEdsPhoneBaseView):
    """
    Ручка для подтверждения номера телефона по коду из смс
    """
    actionlog_name = 'offer_verify_eds_phone'
    authentication_classes = []
    permission_classes = []
    model_class = Offer

    @property
    def instance(self):
        return self.offer
