from rest_framework import serializers

from intranet.femida.src.api.core.serializers import FemidaSerializer, WorkflowActionsField
from intranet.femida.src.api.problems.serializers import ProblemSerializer
from intranet.femida.src.api.users.serializers import UserSerializer
from intranet.femida.src.problems.models import Preset
from intranet.femida.src.problems.helpers import add_active_complaints_count, get_problems_stats
from intranet.femida.src.problems.presets.workflow import PresetWorkflow


class PresetSerializer(serializers.ModelSerializer):

    created_by = UserSerializer()
    # TODO: Выпилить problems_info, когда фронт перейдет на problems_count и categories
    problems_info = serializers.SerializerMethodField()
    problems_count = serializers.SerializerMethodField()
    categories = serializers.SerializerMethodField()

    def get_problems_info(self, obj):
        preset_problem_ids_map = self.context.get('preset_problem_ids_map', {})
        problem_categories_map = self.context.get('problem_categories_map', {})
        problem_ids = preset_problem_ids_map.get(obj.id, [])

        category_names = set()
        for problem_id in problem_ids:
            for category in problem_categories_map.get(problem_id, []):
                category_names.add(category['name'])

        return {
            'count': len(problem_ids),
            'categories': category_names,
        }

    def get_problems_count(self, obj):
        preset_problem_ids_map = self.context.get('preset_problem_ids_map', {})
        return len(preset_problem_ids_map.get(obj.id, []))

    def get_categories(self, obj):
        preset_problem_ids_map = self.context.get('preset_problem_ids_map', {})

        problem_categories_map = self.context.get('problem_categories_map', {})
        problem_ids = preset_problem_ids_map.get(obj.id, [])

        categories_map = {}
        for problem_id in problem_ids:
            for category in problem_categories_map.get(problem_id, []):
                categories_map.setdefault(category['id'], category)
        return list(categories_map.values())

    class Meta:
        model = Preset
        fields = (
            'id',
            'name',
            'created_by',
            'problems_info',
            'problems_count',
            'categories',
        )


class PresetSerializerWithProblems(FemidaSerializer):

    problems = serializers.SerializerMethodField()
    actions = WorkflowActionsField(PresetWorkflow)
    created_by = UserSerializer()

    def get_problems(self, obj):
        problems_qs = (
            obj.problems
            .filter(deleted__isnull=False)
            .prefetch_related('fans')
            .order_by('preset_links__position')
        )
        problems_qs = add_active_complaints_count(problems_qs)
        problems = list(problems_qs)

        user = self.context.get('user')
        if user:
            for problem in problems:
                problem.is_favorite = user in problem.fans.all()

        return ProblemSerializer(
            instance=problems,
            many=True,
            context={
                'problems_stats': get_problems_stats([p.id for p in problems]),
            },
        ).data

    class Meta:
        model = Preset
        fields = (
            'id',
            'name',
            'problems',
            'created_by',
            'actions',
        )


class PresetFilterFormSerializer(serializers.Serializer):

    q = serializers.CharField()
    created_by = serializers.CharField()
