from collections import defaultdict

from rest_framework.permissions import IsAuthenticated

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.core.shortcuts import get_object_or_40x
from intranet.femida.src.api.core.forms import AddProblemsFromPresetForm
from intranet.femida.src.api.core.views import (
    WorkflowViewMixin,
    WorkflowView,
    QueryParamsFormView,
)
from intranet.femida.src.api.problems.permissions import ProblemPermission
from intranet.femida.src.problems.models import Problem, Preset, PresetProblem
from intranet.femida.src.problems.presets.workflow import PresetWorkflow

from . import serializers
from . import forms


from intranet.femida.src.api.core.views import BaseView


class PresetListCreateView(BaseView):

    model_class = Preset
    list_item_serializer_class = serializers.PresetSerializer
    detail_serializer_class = serializers.PresetSerializer
    validator_class = forms.PresetForm
    permission_classes = [
        IsAuthenticated,
        ProblemPermission,
    ]

    def get_queryset(self):
        return self.model_class.objects.select_related('created_by')

    def filter_queryset(self, queryset):
        params = self.request.query_params
        filter_form = forms.PresetFilterForm(data=params)
        self.validate(filter_form)

        filter_params = filter_form.cleaned_data
        if filter_params['created_by']:
            queryset = queryset.filter(created_by=filter_params['created_by'])
        if filter_params['q']:
            queryset = queryset.filter(name__icontains=filter_params['q'])
        return queryset

    def get(self, request, *args, **kwargs):
        """
        Список пресетов
        """
        return self.list(request, *args, **kwargs)

    def get_list_item_serializer_context(self):
        preset_problem_ids_map = defaultdict(list)
        problem_categories_map = defaultdict(list)

        preset_ids = [p.id for p in self.page]
        preset_problem_qs = PresetProblem.objects.filter(preset_id__in=preset_ids)

        for preset_id, problem_id in preset_problem_qs.values_list('preset_id', 'problem_id'):
            preset_problem_ids_map[preset_id].append(problem_id)

        problem_category_qs = (
            Problem.categories.through.objects
            .filter(problem_id__in=preset_problem_qs.values('problem_id'))
            .values_list('problem_id', 'category_id', 'category__name')
        )
        for problem_id, category_id, category_name in problem_category_qs:
            problem_categories_map[problem_id].append({
                'id': category_id,
                'name': category_name,
            })

        return {
            'preset_problem_ids_map': preset_problem_ids_map,
            'problem_categories_map': problem_categories_map,
        }

    @action_logged('preset_create')
    def post(self, request, *args, **kwargs):
        """
        Создание пресета
        """
        return self.create(request, *args, **kwargs)

    def perform_create(self, data):
        data['created_by'] = self.request.user
        preset = Preset.objects.create(**data)
        return preset


class PresetFilterFormView(QueryParamsFormView):

    model_class = Preset
    validator_class = forms.PresetFilterForm
    form_serializer_class = serializers.PresetFilterFormSerializer
    permission_classes = [
        IsAuthenticated,
        ProblemPermission,
    ]


class PresetView(WorkflowViewMixin, BaseView):
    """
    Редактирование пресета
    """
    model_class = Preset
    workflow_class = PresetWorkflow
    detail_serializer_class = serializers.PresetSerializerWithProblems
    validator_class = forms.PresetForm
    permission_classes = [
        IsAuthenticated,
        ProblemPermission,
    ]

    def get_queryset(self):
        return self.model_class.objects.prefetch_related(
            'problems',
            'problems__created_by',
            'problems__categories',
        )

    def get(self, request, *args, **kwargs):
        """
        Один пресет
        """
        return self.retrieve(request, *args, **kwargs)

    @action_logged('preset_update')
    def patch(self, request, *args, **kwargs):
        """
        Редактирование пресета
        """
        return self.perform_action('update')


class PresetProblemView(WorkflowViewMixin, BaseView):

    workflow_class = PresetWorkflow
    model_class = Preset
    permission_classes = [
        IsAuthenticated,
        ProblemPermission,
    ]

    @action_logged('preset_problem_add')
    def put(self, request, *args, **kwargs):
        """
        Привязывание задачи к пресету
        """
        problem = get_object_or_40x(Problem, pk=kwargs.get('problem_id'))
        return self.perform_action(
            action_name='problem_add',
            problem=problem,
        )

    @action_logged('preset_problem_remove')
    def delete(self, request, *args, **kwargs):
        """
        Отвязывание задачи от пресета
        """
        problem = get_object_or_40x(Problem, pk=kwargs.get('problem_id'))
        return self.perform_action(
            action_name='problem_remove',
            problem=problem,
        )


class PresetReorderView(WorkflowViewMixin, BaseView):

    model_class = Preset
    workflow_class = PresetWorkflow
    detail_serializer_class = serializers.PresetSerializerWithProblems
    permission_classes = [
        IsAuthenticated,
        ProblemPermission,
    ]

    @action_logged('preset_reorder')
    def post(self, request, *args, **kwargs):
        """
        Изменение порядка задач в пресете
        """
        problem_ids = request.data.get('problem_ids')
        return self.perform_action(
            action_name='reorder',
            problem_ids=problem_ids,
        )


class PresetAddProblemsFromPresetView(WorkflowView):

    model_class = Preset
    validator_class = AddProblemsFromPresetForm
    workflow_class = PresetWorkflow
    action_name = 'add_problems_from_preset'
    actionlog_name = 'preset_add_problems_from_preset'
    detail_serializer_class = serializers.PresetSerializerWithProblems
    permission_classes = [
        IsAuthenticated,
        ProblemPermission,
    ]
