from rest_framework import serializers

from intranet.femida.src.api.core.fields import SafeWikiFormattedField
from intranet.femida.src.api.core.serializers import (
    AwareSerializerMixin,
    FemidaSerializer,
    IdNameSerializer,
)
from intranet.femida.src.api.users.serializers import UserSerializer
from intranet.femida.src.interviews.models import Assignment
from intranet.femida.src.problems.choices import PROBLEM_SORTING_TYPES
from intranet.femida.src.problems.models import Problem


class ProblemSerializer(AwareSerializerMixin, FemidaSerializer):

    actions = serializers.SerializerMethodField()
    categories = IdNameSerializer(many=True)
    formatted_description = SafeWikiFormattedField()
    formatted_solution = SafeWikiFormattedField()
    created_by = UserSerializer()
    is_favorite = serializers.SerializerMethodField()
    stats = serializers.SerializerMethodField()
    active_complaints_count = serializers.SerializerMethodField()

    used_for_interview = serializers.SerializerMethodField()
    used_for_another_interview = serializers.SerializerMethodField()
    used_for_candidate = serializers.SerializerMethodField()
    used_for_preset = serializers.SerializerMethodField()

    def get_actions(self, obj):
        return {
            'update': not obj.deleted,
            'complain': not obj.deleted,
            'add_to_favorite': not obj.deleted and not self.get_is_favorite(obj),
            'remove_from_favorite': not obj.deleted and self.get_is_favorite(obj),
        }

    def get_is_favorite(self, obj):
        if hasattr(obj, 'is_favorite'):
            return obj.is_favorite
        if 'favorite_problem_ids' in self.root.context:
            return obj.id in self.root.context['favorite_problem_ids']
        user = self.root.context.get('user')
        if user:
            return user in obj.fans.all()

    def get_stats(self, obj):
        assert 'problems_stats' in self.root.context
        return self.root.context['problems_stats'][obj.id]

    def get_active_complaints_count(self, obj):
        if hasattr(obj, 'active_complaints_count'):
            return obj.active_complaints_count
        return self.root.context.get('active_complaints_counts', {}).get(obj.id)

    def _get_problems_usage_param(self, obj, context_obj, param):
        usage_info = self.root.context.get('problems_usage_in_context', {})
        if context_obj not in usage_info:
            return
        return usage_info[context_obj].get('problems', {}).get(obj.id, {}).get(param, False)

    def get_used_for_interview(self, obj):
        return self._get_problems_usage_param(obj, 'interview', 'used_for_interview')

    def get_used_for_another_interview(self, obj):
        return self._get_problems_usage_param(obj, 'interview', 'used_for_another_interview')

    def get_used_for_candidate(self, obj):
        return (
            self.get_used_for_interview(obj)
            or self.get_used_for_another_interview(obj)
        )

    def get_used_for_preset(self, obj):
        return self._get_problems_usage_param(obj, 'preset', 'used_for_preset')

    class Meta:
        model = Problem
        fields = (
            'id',
            'actions',
            'summary',
            'solution',
            'formatted_solution',
            'description',
            'formatted_description',
            'categories',
            'created_by',
            'is_favorite',
            'stats',
            'is_deprecated',
            'deleted',
            'original',
            'active_complaints_count',

            'used_for_interview',
            'used_for_another_interview',
            'used_for_candidate',
            'used_for_preset',
        )
        select_related_map = {
            'created_by': ('created_by',),
        }
        prefetch_related_map = {
            'categories': ('categories',),
            'is_favorite': ('fans',),
            'moderators': ('categories__subscribers',),
        }


class ProblemDetailSerializer(ProblemSerializer):

    moderators = serializers.SerializerMethodField()

    def get_moderators(self, obj):
        moderators = set()
        for category in obj.categories.all():
            for subscriber in category.subscribers.all():
                moderators.add(subscriber)
        return UserSerializer(moderators, many=True).data

    class Meta(ProblemSerializer.Meta):
        fields = ProblemSerializer.Meta.fields + (
            'moderators',
        )


class ProblemAssignmentsSerializer(serializers.ModelSerializer):
    class Meta:
        model = Assignment
        fields = (
            'id',
            'comment',
            'formatted_comment',
            'grade',
        )


class ProblemLiteSerializer(serializers.ModelSerializer):

    class Meta:
        model = Problem
        fields = (
            'summary',
        )


class ProblemCreateFormSerializer(serializers.ModelSerializer):

    categories = serializers.SerializerMethodField()

    def get_categories(self, obj):
        return [i.id for i in obj.categories.all()]

    class Meta:
        model = Problem
        fields = (
            'summary',
            'description',
            'solution',
            'categories',
        )


class ProblemUpdateFormSerializer(ProblemCreateFormSerializer):

    class Meta:
        model = Problem
        fields = ProblemCreateFormSerializer.Meta.fields + (
            'is_deprecated',
        )


class ProblemSearchFormSerializer(serializers.Serializer):

    text = serializers.CharField()
    only_favorite = serializers.BooleanField()
    categories = serializers.SerializerMethodField()
    created_by = serializers.CharField()
    exclude_deprecated = serializers.BooleanField()
    sort = serializers.ChoiceField(choices=PROBLEM_SORTING_TYPES)
    preset = serializers.ReadOnlyField(source='preset.id')

    def get_categories(self, obj):
        return [i.id for i in obj.get('categories', [])]
