from django.db.models import Count

from intranet.femida.src.api.core.pagination import JobsPagination
from intranet.femida.src.api.core.permissions import StrictServicePermission
from intranet.femida.src.api.core.views import BaseView
from intranet.femida.src.api.public_professions import serializers
from intranet.femida.src.jobs.public_professions.models import PublicProfession
from intranet.femida.src.publications.models import Publication
from intranet.femida.src.publications.helpers import get_publications_lang


class PublicProfessionBaseView(BaseView):

    model_class = PublicProfession
    authentication_classes = []
    permission_classes = [
        StrictServicePermission('permissions.can_view_external_publications'),
    ]


class PublicProfessionListView(PublicProfessionBaseView):

    pagination_class = JobsPagination
    list_item_serializer_class = serializers.PublicProfessionSerializer

    def get_queryset(self):
        return super().get_queryset().filter(is_active=True)

    def get_list_item_serializer_context(self):
        lang = get_publications_lang()
        publications_count = dict(
            Publication.objects
            .published_external()
            .filter(
                lang=lang,
                vacancy__profession__public_professions__in=self.page,
            )
            .values('vacancy__profession__public_professions__id')
            .annotate(publications_count=Count('id'))
            .values_list('vacancy__profession__public_professions__id', 'publications_count')
        )
        return {
            'publications_count_by_public_profession_id': publications_count,
        }

    def get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)


class PublicProfessionDetailView(PublicProfessionBaseView):

    detail_serializer_class = serializers.PublicProfessionSerializer

    def get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)
