import sform

from typing import Iterable

from django.conf import settings
from django.db.models.query import QuerySet

from intranet.femida.src.api.core.fetchers import query_params_data_getter
from intranet.femida.src.api.core.forms import (
    CityMultipleChoiceField,
    ProfessionMultipleChoiceWithDataField,
    PublicProfessionMultipleChoiceWithDataField,
    PublicServiceMultipleChoiceField,
    SkillMultipleChoiceField,
)
from intranet.femida.src.jobs.public_professions.models import PublicProfession
from intranet.femida.src.publications.choices import (
    PUBLICATION_PRO_LEVELS,
    PUBLICATION_EMPLOYMENT_TYPES,
)
from intranet.femida.src.utils.cache import method_memoize


class ExternalPublicationListFilterForm(sform.SForm):

    default_getter = query_params_data_getter

    cities = CityMultipleChoiceField(to_field_name='slug', exclude_remote=True)
    professions = ProfessionMultipleChoiceWithDataField(to_field_name='slug')
    public_professions = PublicProfessionMultipleChoiceWithDataField(to_field_name='slug')
    services = PublicServiceMultipleChoiceField(to_field_name='slug')
    pro_levels = sform.MultipleChoiceField(choices=PUBLICATION_PRO_LEVELS)
    employment_types = sform.MultipleChoiceField(choices=PUBLICATION_EMPLOYMENT_TYPES)
    skills = SkillMultipleChoiceField()
    text = sform.CharField()

    @method_memoize(seconds=60 * 60, cache_prefix=settings.JOBS_CACHE_PREFIX)
    def structure_as_dict(self, *args, **kwargs):
        # переопределено для кеширования, structure должна быть одна для любых параметров
        return super().structure_as_dict(*args, **kwargs)

    def gen_queryset_with_fields_per_model_facet(self) -> Iterable[tuple[str, str, str, QuerySet]]:
        """
        Возвращает данные только о фасетах, представленных моделями в БД
        """
        for field_name, field in self.fields.items():
            if isinstance(field, sform.ModelChoiceField):
                yield (
                    field_name,
                    field.kwargs.get('to_field_name', 'pk'),
                    getattr(field, 'publications_path', 'publications'),
                    field.queryset,
                )

    def clean_public_professions(self, public_professions):
        """
        Подмена публичных профессий с is_active=false на текущие публичные профессии
        с is_active=true, которые ссылаются на Фемидные профессии, на которые ссылались исходные

              fp1
            /
        pp1 - fp2
            V

        pp1.is_active=False
        pp2 - fp1
        pp3 - fp2

        Публичная профессия: pp1 объединяет две Фемидные профессии: fp1 и fp2.
        pp1 деактивируется. Создаются две новые публичные профессии: pp2 и pp3.
        pp2 ссылается на fp1, pp3 ссылается на fp2.
        Теперь при запросе pp1 проверяется, какие профессии были с ней связаны, и возвращаются все
        их текущие связи: pp2, pp3
        """
        inactive_qs = public_professions.filter(is_active=False)
        public_professions |= (
            PublicProfession.active
            .filter(professions__in=inactive_qs.values('professions'))
        )
        return public_professions.exclude(is_active=False)
