from django.conf import settings
from django.db.models import Prefetch
from rest_framework import serializers

from intranet.femida.src.api.core.serializers import (
    FemidaSerializer,
    IdNameSerializer,
    IdSlugNameSerializer,
    AwareSerializerMixin,
)
from intranet.femida.src.api.services.serializers import PublicServiceLiteSerializer
from intranet.femida.src.core.serializers import MapperField
from intranet.femida.src.jobs.public_professions.models import PublicProfession
from intranet.femida.src.publications.choices import VACANCY_TO_PUBLICATION_PRO_LEVELS_DISPLAY_MAP
from intranet.femida.src.publications.helpers import get_vacancy_employment_types
from intranet.femida.src.publications.models import Publication
from intranet.femida.src.skills.models import Skill
from intranet.femida.src.vacancies.choices import VACANCY_PRO_LEVELS
from intranet.femida.src.vacancies.models import Vacancy


class ProfessionForPublicationSerializer(IdSlugNameSerializer):

    professional_sphere = IdSlugNameSerializer(name_source='localized_name')


class PublicProfessionForPublicationSerializer(IdSlugNameSerializer):

    professional_sphere = IdSlugNameSerializer(source='public_professional_sphere')


class VacancyForPublicationSerializer(serializers.ModelSerializer):

    cities = IdSlugNameSerializer(many=True)
    skills = IdNameSerializer(source='public_skills', many=True)
    pro_level_min_display = serializers.SerializerMethodField()
    pro_level_max_display = MapperField(
        VACANCY_TO_PUBLICATION_PRO_LEVELS_DISPLAY_MAP,
        source='pro_level_max',
    )
    profession = ProfessionForPublicationSerializer(name_source='localized_name')
    public_professions = PublicProfessionForPublicationSerializer(
        source='profession.active_public_professions',
        default=[],
        many=True,
    )
    employment_types = serializers.SerializerMethodField()

    def get_pro_level_min_display(self, obj):
        if obj.pro_level_max == VACANCY_PRO_LEVELS.intern:
            return None
        return VACANCY_TO_PUBLICATION_PRO_LEVELS_DISPLAY_MAP.get(obj.pro_level_min)

    def get_employment_types(self, obj):
        return get_vacancy_employment_types(obj)

    class Meta:
        model = Vacancy
        fields = (
            'id',
            'cities',
            'skills',
            'pro_level_min_display',
            'pro_level_max_display',
            'profession',
            'public_professions',
            'employment_types',
        )


class ExternalPublicationSerializer(AwareSerializerMixin, FemidaSerializer):

    vacancy = VacancyForPublicationSerializer()
    public_service = PublicServiceLiteSerializer()
    form_url = serializers.SerializerMethodField()

    def get_form_url(self, obj):
        if not obj.form_id:
            return None
        return f'https://{settings.FORMS_EXT_HOST}/surveys/{obj.form_id}?publication_id={obj.id}'

    class Meta:
        model = Publication
        fields = (
            'id',
            'form_url',
            'status',
            'vacancy',
            'published_at',
            'public_service',
            'lang',
            'title',
            'short_summary',
            'key_qualifications',
            'additional_requirements',
            'duties',
            'conditions',
            'description',
            'og_image',
        )
        select_related_map = {
            'public_service': ('public_service',),
        }
        prefetch_related_map = {
            'vacancy': (
                Prefetch(
                    lookup='vacancy__skills',
                    queryset=Skill.objects.filter(is_public=True),
                    to_attr='public_skills',
                ),
                Prefetch(
                    lookup='vacancy__profession__public_professions',
                    queryset=PublicProfession.active.select_related('public_professional_sphere'),
                    to_attr='active_public_professions',
                ),
                'vacancy__profession__professional_sphere',
                'vacancy__cities',
            ),
        }


class ExternalPublicationListSerializer(ExternalPublicationSerializer):

    class Meta(ExternalPublicationSerializer.Meta):
        fields = (
            'id',
            'status',
            'vacancy',
            'public_service',
            'is_chief',
            'lang',
            'title',
            'short_summary',
        )
