import waffle

from django.utils.functional import cached_property

from intranet.femida.src.api.core.pagination import ExternalPublicationPagination
from intranet.femida.src.api.core.permissions import StrictServicePermission
from intranet.femida.src.api.core.views import (
    BaseFormViewMixin,
    BaseView,
    CursorPaginationCountListViewMixin,
    unfold_query_params,
)
from intranet.femida.src.api.publications.external import forms, serializers
from intranet.femida.src.publications.choices import (
    PUBLICATION_FACETS,
    PUBLICATION_STATUSES,
    PUBLICATION_TYPES,
)
from intranet.femida.src.publications.counters import FacetCounter
from intranet.femida.src.publications.helpers import get_suitable_publications
from intranet.femida.src.publications.models import Publication
from intranet.femida.src.utils.begemot import MisspellAPI


class ExternalPublicationBaseView(BaseView):

    model_class = Publication
    authentication_classes = []
    permission_classes = [
        StrictServicePermission('permissions.can_view_external_publications'),
    ]
    valid_statuses = (PUBLICATION_STATUSES.published,)

    def get_queryset(self):
        return self.model_class.objects.filter(
            type=PUBLICATION_TYPES.external,
            status__in=self.valid_statuses,
        )


class ExternalPublicationListView(CursorPaginationCountListViewMixin, BaseFormViewMixin,
                                  ExternalPublicationBaseView):

    list_item_serializer_class = serializers.ExternalPublicationListSerializer
    validator_class = forms.ExternalPublicationListFilterForm
    pagination_class = ExternalPublicationPagination

    @cached_property
    def filter_params(self):
        filter_form = self.get_validator_object(self.get_initial_data())
        self.validate(filter_form)
        return filter_form.cleaned_data

    def filter_queryset(self, queryset):
        queryset = get_suitable_publications(queryset, **self.filter_params)
        return queryset

    def do_spellcheck(self, initial_data, user_text):
        enable_spellcheck = (
            user_text
            and not initial_data.get('disable_spellcheck')
            and waffle.switch_is_active('enable_publication_text_search_spellcheck')
        )
        if enable_spellcheck:
            corrected_text, is_corrected = MisspellAPI.get_spellcheck(user_text, local_rules=True)
            return corrected_text, is_corrected
        return user_text, False

    # TODO: удалить метод после релиза VACANCIES-860
    def is_count_required_for_list(self):
        return True

    def get(self, request, *args, **kwargs):
        """
        Ручка списка внешних объявлений
        """
        initial_data = self.get_initial_data()
        user_text = self.filter_params.get('text')
        corrected_text, is_corrected = self.do_spellcheck(initial_data, user_text)
        # изменяем cached_property, это заметят следующие обращения в рамках
        # одного запроса, это нам и нужно
        self.filter_params['text'] = corrected_text

        available_publication_ids = None
        if user_text:
            queryset = get_suitable_publications(
                queryset=self.get_queryset(),
                text=corrected_text,
            )
            available_publication_ids = set(queryset.values_list('id', flat=True))

        response = self.list(request, *args, **kwargs)
        counters = FacetCounter(initial_data, available_publication_ids)
        response.data['counters'] = counters.compute()

        response.data['spellcheck'] = {
            'is_corrected': is_corrected,
            'user_text': user_text if is_corrected else None,
            'corrected_text': corrected_text if is_corrected else None,
        }
        return response

    def get_initial_data(self):
        return unfold_query_params(
            self.request.query_params,
            list_fields=PUBLICATION_FACETS,
        )


class ExternalPublicationDetailView(ExternalPublicationBaseView):

    detail_serializer_class = serializers.ExternalPublicationSerializer
    valid_statuses = (
        PUBLICATION_STATUSES.published,
        PUBLICATION_STATUSES.archived,
    )

    def get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)
