import sform
from django.core.exceptions import ValidationError
from django.db.models import Q

from intranet.femida.src.api.core.fetchers import query_params_data_getter
from intranet.femida.src.api.core.forms import (
    EMPTY_LABEL,
    CityMultipleSuggestField,
    DepartmentSuggestField,
    ServiceMultipleSuggestField,
    SkillMultipleSuggestField,
    ProfessionMultipleChoiceField,
)
from intranet.femida.src.professions.models import Profession
from intranet.femida.src.vacancies import choices
from intranet.femida.src.vacancies.models import PublicationSubscription
from intranet.femida.src.vacancies.subscriptions.controllers import get_subscription_hash


class InternalPublicationListFilterBaseForm(sform.SForm):

    default_getter = query_params_data_getter

    text = sform.CharField()
    external_url = sform.CharField()
    pro_level_min = sform.ChoiceField(
        empty_label=EMPTY_LABEL,
        choices=choices.VACANCY_PRO_LEVELS,
    )
    pro_level_max = sform.ChoiceField(
        empty_label=EMPTY_LABEL,
        choices=choices.VACANCY_PRO_LEVELS,
    )
    professions = ProfessionMultipleChoiceField()
    skills = SkillMultipleSuggestField()
    cities = CityMultipleSuggestField(only_active=False)
    abc_services = ServiceMultipleSuggestField(only_active=False)
    department = DepartmentSuggestField(only_active=False)
    only_active = sform.BooleanField(default=True)
    sort = sform.ChoiceField(
        choices=choices.PUBLICATION_SORTING_TYPES,
        default=choices.PUBLICATION_SORTING_TYPES.created_desc,
    )

    def clean_pro_level_min(self, value):
        return int(value) if value else None

    def clean_pro_level_max(self, pro_level_max):
        lower = self.cleaned_data.get('pro_level_min')
        upper = int(pro_level_max) if pro_level_max else None

        if lower is not None and upper is not None and lower > upper:
            raise ValidationError(
                message='bound_ordering',
                code='bound_ordering',
            )
        return upper


class InternalPublicationListFilterFrontendForm(InternalPublicationListFilterBaseForm):

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        professions = self.fields['professions']
        # Если в применённом фильтре есть удалённые профессии, нужно
        # их корректно отображать в селекте профессий.
        # При этом, сразу игнорируем точно невалидные значения для id
        profession_ids = [
            p for p in self.initial.get('professions', [])
            if str(p).isdigit()
        ]
        professions.queryset = Profession.objects.filter(
            Q(is_active=True)
            | Q(id__in=profession_ids)
        )


class InternalPublicationListFilterForm(InternalPublicationListFilterBaseForm):

    professions = ProfessionMultipleChoiceField(only_active=False)


class PublicationSubscriptionCreateForm(InternalPublicationListFilterForm):

    def clean(self):
        cleaned_data = super().clean()

        cleaned_data.pop('sort', None)
        cleaned_data['created_by'] = self.context['user']

        subscription_hash = get_subscription_hash(cleaned_data)
        is_duplicate = (
            PublicationSubscription.objects
            .filter(sha1=subscription_hash)
            .exists()
        )
        if is_duplicate:
            # Note: На коллизию не проверяем
            raise ValidationError(
                message='duplicate_subscription',
                code='duplicate_subscription',
            )

        cleaned_data['sha1'] = subscription_hash
        return cleaned_data
