import waffle

from rest_framework import serializers

from intranet.femida.src.api.core.fields import SafeWikiFormattedField
from intranet.femida.src.api.core.serializers import (
    FemidaSerializer,
    IdNameSerializer,
    AwareSerializerMixin,
)
from intranet.femida.src.api.professions.serializers import ProfessionLiteSerializer
from intranet.femida.src.api.staff.serializers import DepartmentSerializer
from intranet.femida.src.api.submissions.serializers import (
    SubmissionFormSerializer,
    PublicationAsSubmissionFormSerializer,
)
from intranet.femida.src.api.users.serializers import UserSerializer
from intranet.femida.src.publications.models import Publication
from intranet.femida.src.vacancies import models
from intranet.femida.src.vacancies.choices import PUBLICATION_ACTIVE_STATUS_LIST


class InternalPublicationSerializer(AwareSerializerMixin, FemidaSerializer):

    profession = ProfessionLiteSerializer()
    skills = IdNameSerializer(many=True)
    hiring_manager = UserSerializer()
    recruiters = UserSerializer(many=True)
    department = DepartmentSerializer()
    pro_level_min_display = serializers.ReadOnlyField(source='get_pro_level_min_display')
    pro_level_max_display = serializers.ReadOnlyField(source='get_pro_level_max_display')
    cities = IdNameSerializer(many=True)
    formatted_publication_content = SafeWikiFormattedField()
    submission_forms = serializers.SerializerMethodField()
    publications = PublicationAsSubmissionFormSerializer(many=True)
    abc_services = IdNameSerializer(many=True)
    is_active = serializers.SerializerMethodField()
    actions = serializers.SerializerMethodField()

    def get_submission_forms(self, obj):
        publications = PublicationAsSubmissionFormSerializer(obj.publications.all(), many=True).data
        if not waffle.switch_is_active('disable_submission_forms_in_serializers'):
            publications += SubmissionFormSerializer(obj.submission_forms.all(), many=True).data
        return publications

    def get_is_active(self, obj):
        return obj.status in PUBLICATION_ACTIVE_STATUS_LIST

    def get_actions(self, obj):
        is_active = self.get_is_active(obj)
        is_user_rotating = self.root.context.get('is_user_rotating', False)
        return {
            'recommend': is_active,
            'rotate': is_active and not is_user_rotating,
            'share': is_active,
        }

    class Meta:
        model = models.Vacancy
        fields = (
            'id',
            'status',
            'created',
            'profession',
            'skills',
            'hiring_manager',
            'recruiters',
            'department',
            'pro_level_min_display',
            'pro_level_max_display',
            'cities',
            'publication_title',
            'publication_content',
            'formatted_publication_content',
            'submission_forms',
            'publications',
            'abc_services',
            'is_active',
            'actions',
        )
        select_related_map = {
            'profession': ('profession__professional_sphere',),
            'department': ('department',),
        }
        prefetch_related_map = {
            'skills': ('skills',),
            'hiring_manager': ('memberships__member',),
            'recruiters': ('memberships__member',),
            'cities': ('cities',),
            'submission_forms': ('submission_forms', 'publications'),
            'publications': ('publications',),
            'abc_services': ('abc_services',),
        }


class PublicationSubscriptionSerializer(AwareSerializerMixin, FemidaSerializer):

    actions = serializers.SerializerMethodField()
    created_by = UserSerializer()

    pro_level_min_display = serializers.ReadOnlyField(source='get_pro_level_min_display')
    pro_level_max_display = serializers.ReadOnlyField(source='get_pro_level_max_display')
    professions = IdNameSerializer(name_source='localized_name', many=True)
    skills = IdNameSerializer(many=True)
    cities = IdNameSerializer(many=True)
    abc_services = IdNameSerializer(many=True)
    department = IdNameSerializer('localized_name')

    class Meta:
        model = models.PublicationSubscription
        fields = (
            'id',
            'actions',
            'created_by',
            'created',
            'text',
            'external_url',
            'pro_level_min',
            'pro_level_max',
            'pro_level_min_display',
            'pro_level_max_display',
            'professions',
            'skills',
            'cities',
            'abc_services',
            'department',
            'only_active',
        )
        select_related_map = {
            'created_by': ('created_by',),
            'department': ('department',),
        }
        prefetch_related_map = {
            'skills': ('skills',),
            'cities': ('cities',),
            'abc_services': ('abc_services',),
            'professions': ('professions',),
        }

    def get_actions(self, obj):
        return {
            'delete': self.root.context['user'] == obj.created_by,
        }


class PublicationSerializer(FemidaSerializer):

    class Meta:
        model = Publication
        fields = (
            'id',
            'uuid',
            'status',
            'type',
            'lang',
            'title',
            'short_summary',
        )
