import logging

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.api.core.views import (
    BaseView,
    BaseFormViewMixin,
    unfold_query_params,
)
from intranet.femida.src.api.publications import serializers, forms
from intranet.femida.src.api.publications.permissions import (
    PublicationPermission,
    PublicationSubscriptionPermission,
)
from intranet.femida.src.candidates.rotations.helpers import is_user_rotating
from intranet.femida.src.publications.models import Publication
from intranet.femida.src.vacancies.subscriptions.controllers import create_publication_subscription
from intranet.femida.src.vacancies.helpers import get_suitable_vacancies
from intranet.femida.src.vacancies.models import Vacancy, PublicationSubscription


logger = logging.getLogger(__name__)


class InternalPublicationBaseView(BaseView):
    """
    Базовая вьюха для внутренних объявлений
    """
    model_class = Vacancy
    permission_classes = [PublicationPermission]

    def get_queryset(self):
        """
        Отдаем опубликованные вакансии без проверки доступов.

        Выбор полей, безопасных для отображения всем пользователям, возлагается на сериализатор.
        """
        return self.model_class.unsafe.published()


class InternalPublicationListView(InternalPublicationBaseView):
    """
    Вьюха объединяет в себе 2 ручки:
    1. Ручка фильтров объявлений
    2. Ручка списка/фильтрации объявлений
    """
    list_item_serializer_class = serializers.InternalPublicationSerializer
    validator_class = forms.InternalPublicationListFilterForm

    def get(self, request, *args, **kwargs):
        """
        Ручка списка внутр.объявлений по вакансиям
        """
        return self.list(request, *args, **kwargs)

    def get_initial_data(self):
        return unfold_query_params(
            self.request.query_params,
            list_fields=(
                'skills',
                'cities',
                'abc_services',
                'professions',
            ),
        )

    def get_list_item_serializer_context(self):
        return {
            'is_user_rotating': is_user_rotating(self.request.user),
        }

    def filter_queryset(self, queryset):
        filter_form = self.get_validator_object(self.get_initial_data())
        self.validate(filter_form)
        filter_params = filter_form.cleaned_data

        queryset = get_suitable_vacancies(queryset, **filter_params)

        if filter_params['sort']:
            queryset = queryset.order_by(filter_params['sort'])

        return queryset.distinct()


class InternalPublicationListFilterFormView(InternalPublicationListView, BaseFormViewMixin):

    validator_class = forms.InternalPublicationListFilterFrontendForm


class InternalPublicationDetailView(InternalPublicationBaseView):

    detail_serializer_class = serializers.InternalPublicationSerializer

    def get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)

    def get_detail_serializer_context(self):
        return {
            'is_user_rotating': is_user_rotating(self.request.user),
        }


class PublicationSubscriptionListCreateView(BaseView):

    model_class = PublicationSubscription
    list_item_serializer_class = serializers.PublicationSubscriptionSerializer
    detail_serializer_class = serializers.PublicationSubscriptionSerializer
    validator_class = forms.PublicationSubscriptionCreateForm
    permission_classes = [PublicationPermission]

    def filter_queryset(self, queryset):
        return (
            queryset
            .filter(created_by=self.request.user)
            .order_by('-created')
        )

    def get(self, request, *args, **kwargs):
        """
        Ручка списка сохраненных фильтров пользователя
        """
        return self.list(request, *args, **kwargs)

    @action_logged('publication_subscription_create')
    def post(self, request, *args, **kwargs):
        """
        Ручка создания сохраненного фильтра
        """
        return self.create(request, *args, **kwargs)

    def perform_create(self, data):
        return create_publication_subscription(data)

    def get_validator_context(self):
        return {
            'user': self.request.user,
        }


class PublicationSubscriptionDetailView(BaseView):

    model_class = PublicationSubscription
    detail_serializer_class = serializers.PublicationSubscriptionSerializer
    permission_classes = [PublicationSubscriptionPermission]

    def get(self, request, *args, **kwargs):
        """
        Ручка деталей сохраненного фильтра
        """
        return self.retrieve(request, *args, **kwargs)

    @action_logged('publication_subscription_delete')
    def delete(self, request, *args, **kwargs):
        """
        Ручка удаления сохраненного фильтра
        """
        return self.destroy(request, *args, **kwargs)

    def perform_destroy(self, instance):
        instance.delete()


class PublicationBaseView(BaseView):

    model_class = Publication


class PublicationDetailView(PublicationBaseView):

    authentication_classes = []
    permission_classes = []
    detail_serializer_class = serializers.PublicationSerializer

    def get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)
