from itertools import groupby

from django.db.models import Count, Avg

from intranet.femida.src.interviews.models import Application, Interview, Assignment
from intranet.femida.src.problems.models import Problem

from intranet.femida.src.api.stats import utils


def get_applications_by_month(*args, **kwargs):
    """
    Количество завершенных претенденств по месяцам.
    """
    qs = (
        Application.objects
        .filter(finished__isnull=False)
        .extra(select=utils.month_select('finished'))
        .values('month', 'year')
        .annotate(count=Count('finished'))
        .order_by('-count')
    )
    return list(qs)


def get_interviews_by_state(user=None, *args, **kwargs):
    """
    Количество собеседований по статусам.
    """
    base_qs = Interview.objects.all()
    if user:
        base_qs = base_qs.filter(interviewer_id=user.id)

    qs = (
        base_qs
        .values('state')
        .annotate(count=Count('state'))
        .order_by('-count')
    )
    return list(qs)


def get_interviews_by_grade(user=None, date_from=None, date_to=None,
                            *args, **kwargs):
    """
    Количество завершенных собеседований по грейдам.
    """
    base_qs = Interview.objects.filter(state='finished')

    if user:
        base_qs = base_qs.filter(interviewer_id=user.id)
    if date_from:
        base_qs = base_qs.filter(finished__gte=date_from)
    if date_to:
        base_qs = base_qs.filter(finished__lt=date_to)

    qs = (
        base_qs
        .values('grade')
        .annotate(count=Count('grade'))
        .order_by('grade')
    )
    print(qs.query)
    return list(qs)


def get_interviews_by_month(user=None, *args, **kwargs):
    """
    Количество завершенных собеседований и средняя оценка по месяцам.
    """
    base_qs = Interview.objects.filter(state='finished')

    if user:
        base_qs = base_qs.filter(interviewer_id=user.id)

    qs = (
        base_qs
        .extra(select=utils.month_select('finished'))
        .values('month', 'year')
        .annotate(avg=Avg('grade'))
        .annotate(count=Count('finished'))
        .order_by('-count')
    )
    return list(qs)


def get_interviews_by_interviewer(date_from=None, date_to=None,
                                  *args, **kwargs):
    """
    Количество завершенных собеседований и средняя оценка по людям.
    """
    base_qs = Interview.objects.filter(state='finished')

    if date_from:
        base_qs = base_qs.filter(finished__gte=date_from)
    if date_to:
        base_qs = base_qs.filter(finished__lt=date_to)

    qs = (
        base_qs
        .values('interviewer__username')
        .annotate(avg_grade=Avg('grade'))
        .annotate(count=Count('interviewer_id'))
        .order_by('-count')
    )
    return list(qs)


def get_interviews_by_application(*args, **kwargs):
    """
    Список грейдов собеседований для каждого завершенного претенденства.
    """
    result = []
    qs = (
        Interview.objects
        .filter(state='finished', application__finished__isnull=False)
        .values('application_id', 'grade')
        .order_by('application_id')
    )
    for application_id, grades in groupby(qs, lambda x: x['application_id']):
        result.append({
            'application_id': application_id,
            'grades': [i['grade'] for i in grades],
        })
    return result


def get_problems_by_month(user=None, *args, **kwargs):
    """
    Количество созданных задач по месяцам.
    """
    base_qs = Problem.objects.all()

    if user:
        base_qs = base_qs.filter(created_by_id=user.id)

    qs = (
        base_qs
        .extra(select=utils.month_select('created'))
        .values('month', 'year')
        .annotate(count=Count('created'))
        .order_by('-count')
    )
    return list(qs)


def get_assignments_by_problem(user=None, date_from=None, date_to=None,
                               *args, **kwargs):
    """
    Количество использований задач.
    """
    base_qs = (
        Assignment.objects
        .filter(grade__isnull=False, interview__state='finished')
    )

    if user:
        base_qs = base_qs.filter(interview__interviewer_id=user.id)
    if date_from:
        base_qs = base_qs.filter(created__gte=date_from)
    if date_to:
        base_qs = base_qs.filter(created__lt=date_to)

    qs = (
        base_qs
        .values('problem_id')
        .annotate(count=Count('problem_id'))
        .order_by('-count')
    )
    return list(qs)
