import logging

from django.contrib.auth import get_user_model

from intranet.femida.src.api.core.views import BaseView, WorkflowView, BaseFormViewMixin
from intranet.femida.src.candidates.choices import (
    SUBMISSION_STATUSES,
    SUBMISSION_SOURCES,
    EXTERNAL_SUBMISSION_SOURCES,
)
from intranet.femida.src.candidates.models import CandidateSubmission
from intranet.femida.src.candidates.submissions.helpers import filter_submissions_by_recruiter
from intranet.femida.src.candidates.submissions.workflow import SubmissionWorkflow
from intranet.femida.src.communications.choices import MESSAGE_TEMPLATE_TYPES
from intranet.femida.src.communications.models import MessageTemplate

from . import serializers, forms


logger = logging.getLogger(__name__)

User = get_user_model()


class SubmissionListFilterFormView(BaseFormViewMixin, BaseView):
    """
    Вьюха объединяет в себе 2 ручки:
    1. Ручка формы фильтров отликов
    2. Ручка списка/фильтрации откликов
    """
    model_class = CandidateSubmission
    list_item_serializer_class = serializers.SubmissionSerializer
    validator_class = forms.SubmissionListFilterForm

    def get_initial_data(self):
        return self.request.query_params

    def get_queryset(self):
        return (
            self.model_class.objects
            .exclude(
                # От всех скрываем черновики-ротации
                status=SUBMISSION_STATUSES.draft,
                source=SUBMISSION_SOURCES.rotation,
            )
            .order_by('-created')
        )

    def filter_queryset(self, queryset):
        # От некоординаторов всегда скрываем черновики
        if not self.request.user.is_reference_coordinator:
            queryset = queryset.exclude(status=SUBMISSION_STATUSES.draft)

        filter_form = self.get_validator_object(self.request.query_params)
        self.validate(filter_form)
        params = filter_form.cleaned_data

        simple_filter_fields = {
            'status',
            'source',
            'reference__status',
        }
        if params.get('source') == SUBMISSION_SOURCES.form:
            queryset = queryset.filter(source__in=EXTERNAL_SUBMISSION_SOURCES._db_values)
            simple_filter_fields.discard('source')

        queryset = queryset.filter(**{i: params[i] for i in simple_filter_fields if params.get(i)})

        # Позволяем рекрутерам смотреть все офферы или только свои.
        # Также позволяем смотреть офферы другого рекрутера.
        # На фильтрацию по правам это не влияет.
        if params['recruiter']:
            queryset = filter_submissions_by_recruiter(queryset, params['recruiter'])

        return queryset

    def get(self, request, *args, **kwargs):
        """
        Список откликов
        """
        return self.list(request, *args, **kwargs)


class SubmissionDetailView(BaseView):

    model_class = CandidateSubmission
    detail_serializer_class = serializers.SubmissionDetailSerializer

    def get(self, request, *args, **kwargs):
        """
        Один отклик вместе с возможными дубликатами.
        """
        return self.retrieve(request, *args, **kwargs)


class SubmissionWorkflowView(WorkflowView):

    workflow_class = SubmissionWorkflow
    model_class = CandidateSubmission
    detail_serializer_class = serializers.SubmissionDetailSerializer

    def get_queryset(self):
        return (
            self.model_class.objects
            .select_related(
                'form',
                'publication',
                'reference',
            )
        )

    @property
    def actionlog_name(self):
        return 'submission_%s' % self.action_name


class SubmissionHandleView(BaseFormViewMixin, SubmissionWorkflowView):

    action_name = 'handle'
    validator_class = forms.SubmissionHandleForm

    def get_validator_context(self):
        return {
            'submission': self.get_object(),
        }


class SubmissionRejectView(SubmissionWorkflowView):

    action_name = 'reject'


class ReferenceActionViewBase(BaseFormViewMixin, SubmissionWorkflowView):

    validator_class = forms.ReferenceActionForm
    message_template_name = None

    def get_initial_data(self):
        message_template = (
            MessageTemplate.objects
            .filter(
                type=MESSAGE_TEMPLATE_TYPES.reference,
                name=self.message_template_name,
            )
            .first()
        )
        if message_template:
            comment = message_template.text
        else:
            logger.warning(
                'There is no message template with type `%s` and name `%s`',
                MESSAGE_TEMPLATE_TYPES.reference,
                self.message_template_name,
            )
            comment = ''
        return {
            'comment': comment,
        }


class ReferenceApproveView(ReferenceActionViewBase):

    action_name = 'approve_reference'
    message_template_name = 'Принять'


class ReferenceApproveWithoutBenefitsView(ReferenceActionViewBase):

    action_name = 'approve_reference_without_benefits'
    message_template_name = 'Принять без вознаграждения'


class ReferenceRejectView(ReferenceActionViewBase):

    action_name = 'reject_reference'
    message_template_name = 'Отклонить'
