import re

from django.db.models import Q
from django.db.models.functions import Length
from rest_framework import status
from rest_framework.response import Response

from intranet.femida.src.api.core.views import BaseView, ResponseError
from intranet.femida.src.applications.models import Application
from intranet.femida.src.candidates.models import Candidate
from intranet.femida.src.core.models import City, Tag
from intranet.femida.src.offers.models import Position
from intranet.femida.src.problems.helpers import has_access_to_problems
from intranet.femida.src.problems.models import Category, Preset, Problem
from intranet.femida.src.professions.models import Profession
from intranet.femida.src.skills.models import Skill
from intranet.femida.src.vacancies.choices import (
    ACTIVE_VACANCY_STATUSES,
    PUBLICATION_ACTIVE_STATUS_LIST,
)
from intranet.femida.src.vacancies.models import Vacancy
from intranet.femida.src.utils.translation import get_name_field, get_localized_name_field


class SuggestView(BaseView):

    def _get_suggest_data(self, queryset, search_fields, extra_fields=(), order=None):
        """
        :param queryset: QuerySet, по которому делаем саджест
        :param search_fields: Поля, по которым ищем
        :param extra_fields: Поля, которые дополнительно нужно отдать
        :param order: По каким полям сортируем
        :return:
        """
        q = self.request.query_params.get('q', '')

        if order:
            queryset = queryset.order_by(*order)

        def construct_search(field_name):
            """
            Как в админке
            """
            if field_name.startswith('^'):
                return '%s__istartswith' % field_name[1:]
            elif field_name.startswith('='):
                return '%s__iexact' % field_name[1:]
            elif field_name.startswith('@'):
                return '%s__search' % field_name[1:]
            else:
                return '%s__icontains' % field_name

        db_query = Q()
        for field_name in search_fields:
            db_query |= Q(**{construct_search(field_name): q})
        search_fields = [f.strip('^=@') for f in search_fields]
        values_fields = set(search_fields) | set(extra_fields)
        data = list(queryset.filter(db_query).values(*values_fields)[:5])
        return data

    def get_candidate_data(self):
        return self._get_suggest_data(
            queryset=Candidate.objects.all(),
            search_fields=('id', 'first_name', 'last_name'),
        )

    def get_vacancy_data(self):
        statuses = self.request.query_params.get('statuses')
        filter_data = {}
        if not statuses:
            filter_data['status__in'] = ACTIVE_VACANCY_STATUSES._db_values
        elif statuses != '*':
            filter_data['status__in'] = statuses.split(',')
        data = self._get_suggest_data(
            queryset=Vacancy.objects.filter(**filter_data),
            search_fields=('id', 'name'),
            extra_fields=('status',),
        )
        return data

    def get_publication_data(self):
        qs = (
            Vacancy.unsafe
            .published()
            .filter(status__in=PUBLICATION_ACTIVE_STATUS_LIST)
        )
        data = self._get_suggest_data(
            queryset=qs,
            search_fields=('id', 'publication_title', 'department__name'),
        )
        for item in data:
            item['department_name'] = item.pop('department__name')
        return data

    def get_application_data(self):
        data = self._get_suggest_data(
            queryset=Application.objects.all(),
            search_fields=(
                'id',
                'candidate__first_name',
                'candidate__last_name',
                'vacancy__name',
            ),
        )
        for v in data:
            v['name'] = (
                'Application #{id} - {vacancy__name}, '
                '{candidate__first_name} {candidate__last_name}'
            ).format(**v)
        return data

    def get_skill_data(self):
        return self._get_suggest_data(
            queryset=Skill.objects.annotate(name_len=Length('name')),
            search_fields=('id', 'name'),
            order=('name_len',),
        )

    def get_profession_data(self):
        localized_name_field = get_localized_name_field()

        data = self._get_suggest_data(
            queryset=Profession.active.all(),
            search_fields=('id', 'name', 'name_en'),
            extra_fields=(
                'professional_sphere_id',
                'professional_sphere__name',
                'professional_sphere__name_en'
            ),
        )
        data = [
            {
                'id': v['id'],
                'name': v[localized_name_field],
                'professional_sphere_id': v['professional_sphere_id'],
                'professional_sphere_name': v[f'professional_sphere__{localized_name_field}'],
            }
            for v in data
        ]
        return data

    def get_position_data(self):
        name_field = get_name_field()

        data = self._get_suggest_data(
            queryset=Position.objects.annotate(name_len=Length('name_ru')).alive(),
            search_fields=('id', 'staff_id', 'name_ru', 'name_en'),
            order=('name_len',),
        )
        data = [{'id': v['id'], 'name': v[name_field]} for v in data]
        return data

    def get_city_data(self):
        name_field = get_name_field()

        data = self._get_suggest_data(
            queryset=City.active.all(),
            search_fields=('id', 'name_ru', 'name_en'),
        )
        data = [{'id': v['id'], 'name': v[name_field]} for v in data]
        return data

    def get_category_data(self):
        return self._get_suggest_data(
            queryset=Category.objects.all(),
            search_fields=('name',),
        )

    def get_problem_data(self):
        if not has_access_to_problems(self.request.user):
            return []
        q = self.request.query_params.get('q', '')
        search_fields = ('id', 'summary')

        extra_fields = (
            'created_by__first_name',
            'created_by__last_name',
            'created_by__username',
        )
        url_rgx = re.compile(r'/problems/(?P<id>\d+)/?')
        rgx_res = url_rgx.search(q)
        if rgx_res:
            data = list(
                Problem.alive_objects
                .filter(id=rgx_res.groupdict().get('id'))
                .values(*(search_fields + extra_fields))
            )
        else:
            data = self._get_suggest_data(
                queryset=Problem.alive_objects.all(),
                search_fields=search_fields,
                extra_fields=extra_fields,
            )
        for item in data:
            item['created_by'] = {
                f.replace('created_by__', ''): item.pop(f)
                for f in extra_fields
            }
        return data

    def get_preset_data(self):
        extra_fields = (
            'created_by__first_name',
            'created_by__last_name',
            'created_by__username',
        )
        data = self._get_suggest_data(
            queryset=Preset.objects.all(),
            search_fields=('id', '=created_by__username', 'name'),
            extra_fields=extra_fields,
        )
        for item in data:
            item['created_by'] = {
                f.replace('created_by__', ''): item.pop(f)
                for f in extra_fields
            }
        return data

    def get_tag_data(self):
        return self._get_suggest_data(
            queryset=Tag.objects.all(),
            search_fields=('id', 'name'),
        )

    def get(self, request, *args, **kwargs):
        """
        Универсальный саджест по сущностям из Фемиды
        """
        types = request.query_params.get('types')

        getter = getattr(self, 'get_%s_data' % types, None)
        if getter is None:
            raise ResponseError(
                data={
                    'types': 'Invalid type',
                },
                status=status.HTTP_400_BAD_REQUEST,
            )
        data = getter()
        return Response(data, status=status.HTTP_200_OK)
