import logging

from django.core.exceptions import ObjectDoesNotExist, MultipleObjectsReturned
from django.utils.functional import cached_property
from rest_framework.response import Response

from intranet.femida.src.api.core.views import BaseView, ResponseError
from intranet.femida.src.monitoring.utils import alarm
from intranet.femida.src.permissions.helpers import get_manager
from intranet.femida.src.startrek.tasks import tracker_trigger_task


logger = logging.getLogger(__name__)


class TrackerTriggerBaseView(BaseView):
    """
    Базовая вьюха для запросов от триггеров Трекера.
    При любых обстоятельствах отвечает 200,
    если что-то пошло не так, просто пишет об этом в лог.

    Принимает только POST-запросы, в данных ожидается `issue_key`
    - ключ тикета. По данному ключу пытается найти объект модели `model_class`.
    Если объект найден, вызывается заданный celery-task `task` с одним аргументом
    - id найденного объекта.

    Наследники должны переопределить как минимум 3 атрибута:
    model_class - класс модели, для которой нужно выполнить действие
    issue_key_field - поле модели с ключом тикета,
      по которому нужно найти соответствующий объект
    task - celery-таск, который будет запущен для найденного объекта
    """
    permission_classes = []
    authentication_classes = []
    issue_key_field = None
    task = None
    valid_statuses = None
    is_retriable = True

    def warning(self, text, *args):
        """
        Пишет в лог сообщение об ошибке и сразу возвращает ответ
        """
        text = text.format(
            model=self.model_class.__name__,
            issue_key=self.request.data.get('issue_key'),
        )
        logger.warning('TRACKER TRIGGER: ' + text, *args)
        raise ResponseError()

    @cached_property
    def issue_key(self):
        key = self.request.data.get('issue_key')
        if key:
            return key
        self.warning('missing "issue_key" parameter')

    def get_queryset(self):
        # Всегда получаем `unsafe` объекты для Трекера,
        # потому что он ходит без авторизации
        return get_manager(self.model_class, unsafe=True).all()

    def get_object_in_valid_status(self, issue_key):
        qs = self.get_queryset().filter(**{self.issue_key_field: issue_key})

        if len(qs) == 0:
            raise ObjectDoesNotExist

        instances = qs
        if self.valid_statuses:
            instances = [i for i in qs if i.status in self.valid_statuses]

        if len(instances) > 1:
            raise MultipleObjectsReturned(
                f'Found multiple {self.model_class.__name__} objects with issue_key={issue_key}'
            )

        return instances[0] if instances else None

    def get_object(self):
        if hasattr(self, '_instance'):
            return self._instance
        try:
            instance = self.get_object_in_valid_status(issue_key=self.issue_key)
        except ObjectDoesNotExist:
            if self.is_retriable:
                tracker_trigger_task.delay(
                    view_name=self.__class__.__name__,
                    module=self.__class__.__module__,
                    issue_key=self.issue_key,
                )
            self.warning('{model} with issue_key={issue_key} was not found')
        except MultipleObjectsReturned:
            alarm(
                f'Found multiple {self.model_class.__name__} objects with '
                f'issue_key={self.issue_key}'
            )
            raise ResponseError()

        if not instance:
            self.warning('{model} with issue_key={issue_key} has invalid status')

        self._instance = instance
        return self._instance

    def post(self, request, *args, **kwargs):
        instance = self.get_object()
        self.task.delay(instance.id)
        return Response()
