import waffle
from rest_framework import serializers
from rest_framework.fields import empty
from rest_framework.utils import html

from intranet.femida.src.api.core.serializers import LocalizedField
from intranet.femida.src.candidates.workflow import CandidateWorkflow


class UserSerializer(serializers.Serializer):

    id = serializers.ReadOnlyField()
    username = serializers.ReadOnlyField()
    login = serializers.ReadOnlyField(source='username')
    firstname = LocalizedField(source='first_name')
    lastname = LocalizedField(source='last_name')
    fullname = LocalizedField(source='get_full_name')
    gender = serializers.ReadOnlyField()
    is_dismissed = serializers.ReadOnlyField()

    def get_value(self, dictionary):
        """
        Почему-то дефолтный get_value в случае с form-data не парсит
        этот сериалайзер

        TODO: Проверить актуально ли это сейчас и где у нас вообще form-data
        """
        value = dictionary.get(self.field_name, empty)
        if html.is_html_input(dictionary):
            if isinstance(value, list) and len(value):
                value = value[0]
        return value


class AuthUserSerializer(UserSerializer):

    staff_id = serializers.ReadOnlyField()
    work_phone = serializers.ReadOnlyField()
    tz = serializers.ReadOnlyField(source='timezone')

    actions = serializers.SerializerMethodField()

    is_recruiter = serializers.ReadOnlyField()
    is_recruiter_assessor = serializers.ReadOnlyField()
    is_recruiting_manager = serializers.ReadOnlyField()
    is_moderator = serializers.ReadOnlyField()
    is_interviewer = serializers.SerializerMethodField()

    def get_actions(self, obj):
        return {
            'candidate_create': (
                CandidateWorkflow(instance=None, user=obj).get_action('create').is_available()
            ),
            'vacancy_create': True,
            'vacancy_group_create': True,
            'problem_create': True,
            'preset_create': True,
            'publication_subscription_create': (
                waffle.switch_is_active('enable_publication_subscriptions')
            ),
            'category_subscribe': obj.is_moderator,
        }

    def get_is_interviewer(self, obj):
        return True


class InterviewerSerializer(UserSerializer):

    interviews_finished_2weeks_count = serializers.ReadOnlyField()
    interviews_planned_count = serializers.ReadOnlyField()
    interviews_total_count = serializers.ReadOnlyField()


class UserWithDepartmentSerializer(UserSerializer):

    department_name = LocalizedField(source='department.name')
