import sform
import waffle

from django.core.exceptions import ValidationError
from django.db.models import Count, Q
from django.utils.functional import cached_property

from intranet.femida.src.api.core.fetchers import query_params_data_getter
from intranet.femida.src.api.core.forms import (
    UserSuggestField,
    UserMultipleSuggestField,
    SkillSuggestField,
    CityMultipleSuggestField,
    ProfessionChoiceField,
    DepartmentSuggestField,
    ValueStreamSuggestField,
    GeographySuggestField,
    ServiceMultipleSuggestField,
    SkillMultipleSuggestField,
    validate_recruiters,
    ConditionallyRequiredFieldsMixin,
    EMPTY_LABEL,
    PG_MAX_INTEGER_VALUE,
    PositiveIntegerField,
    BaseMetaForm,
    OfficeMultipleChoiceField,
    LocationMultipleSuggestField,
    WorkModeMultipleChoiceField,
)
from intranet.femida.src.api.vacancies.fields import (
    HeadLevelField,
    HiringManagerLevelField,
    ResponsibleLevelField,
    is_role_field,
)
from intranet.femida.src.core.choices import get_partial_choices
from intranet.femida.src.core.models import Currency
from intranet.femida.src.interviews.choices import CLOSED_APPLICATION_RESOLUTIONS_MANAGEABLE
from intranet.femida.src.professions.models import ProfessionalSphere
from intranet.femida.src.staff.choices import GEOGRAPHY_KINDS
from intranet.femida.src.vacancies import choices
from intranet.femida.src.vacancies.helpers import get_vacancy_group_label
from intranet.femida.src.vacancies.models import VacancyGroup, Vacancy, ValueStream, Geography
from intranet.femida.src.vacancies.startrek.memberships import IssueMembership


# Типы вакансий, которые отдаем на фронт
FORM_VACANCY_TYPES = get_partial_choices(
    choices.VACANCY_TYPES,
    choices.VACANCY_TYPES.new,
    choices.VACANCY_TYPES.replacement,
    choices.VACANCY_TYPES.internship,
)


def clean_budget_position_id(bp_id, exclude_ids=None):
    exclude_ids = exclude_ids or []
    queryset = (
        Vacancy.unsafe
        .filter(budget_position_id=bp_id)
        .exclude(status=choices.VACANCY_STATUSES.closed)
        .exclude(id__in=exclude_ids)
    )

    if queryset.exists():
        raise ValidationError(
            message='budget_position_already_exists',
            code='budget_position_already_exists',
        )

    return bp_id


class VacancySkillForm(sform.SForm):
    skill = SkillSuggestField()
    is_required = sform.BooleanField()


class VacancyBaseForm(sform.SForm):
    """
    Базовая форма создания/редактирования вакансии
    """
    _exclude_fields = ()
    name = ResponsibleLevelField(sform.CharField, state=sform.REQUIRED, max_length=255)
    hiring_manager = HeadLevelField(UserSuggestField, state=sform.REQUIRED)
    department = DepartmentSuggestField(state=sform.REQUIRED)
    locations = ResponsibleLevelField(
        LocationMultipleSuggestField,
        create_missing=True,
    )
    professional_sphere = sform.ModelChoiceField(
        queryset=ProfessionalSphere
            .active
            .annotate(professions_count=Count('professions', filter=Q(professions__is_active=True)))
            .filter(professions_count__gte=1),
        label_extractor=lambda x: x.localized_name,
        state=sform.REQUIRED,
    )
    profession = ProfessionChoiceField(empty_label=EMPTY_LABEL, state=sform.REQUIRED)
    pro_level_min = ResponsibleLevelField(
        sform.ChoiceField,
        empty_label=EMPTY_LABEL,
        choices=choices.VACANCY_PRO_LEVELS,
    )
    pro_level_max = ResponsibleLevelField(
        sform.ChoiceField,
        empty_label=EMPTY_LABEL,
        choices=choices.VACANCY_PRO_LEVELS,
        state=sform.REQUIRED,
    )

    def clean_pro_level_min(self, value):
        return int(value) if value else None

    def clean_pro_level_max(self, pro_level_max):
        lower = self.cleaned_data.get('pro_level_min')
        upper = int(pro_level_max) if pro_level_max else None

        if lower is not None and upper is not None and lower > upper:
            raise ValidationError(
                message='bound_ordering',
                code='bound_ordering',
            )
        return upper

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        for field in set(self._exclude_fields):
            assert field in self.fields
            self.fields.pop(field)


class IssueOnlyFieldsMixin(sform.SForm):
    """
    Миксин, который содержит общие поля, которые нужны только для прокидывания их в ST тикет
    """
    wage_system = sform.ChoiceField(choices=choices.WAGE_SYSTEM_TYPES, state=sform.REQUIRED)
    max_salary = sform.IntegerField()
    currency = sform.ModelChoiceField(
        queryset=Currency.active.order_by('id'),
        label_extractor=lambda x: x.code,
    )
    followers = UserMultipleSuggestField()


class VacancyChooseTypeForm(sform.SForm):

    default_getter = query_params_data_getter

    type = sform.ChoiceField(
        choices=FORM_VACANCY_TYPES,
        state=sform.REQUIRED,
        empty_label=EMPTY_LABEL,
    )


class VacancyCreateBaseForm(IssueOnlyFieldsMixin, VacancyBaseForm):

    default_getter = query_params_data_getter

    head = UserSuggestField(state=sform.READONLY)
    type = sform.ChoiceField(choices=FORM_VACANCY_TYPES, state=sform.REQUIRED)
    value_stream = ResponsibleLevelField(
        sform.ModelChoiceField,
        queryset=ValueStream.objects.filter(
            is_active=True,
            oebs_product_id__isnull=False,
        ),
        label_extractor=lambda x: x.name,
        state=sform.REQUIRED,
    )
    work_mode = ResponsibleLevelField(
        WorkModeMultipleChoiceField,
        state=sform.REQUIRED,
    )

    access = UserMultipleSuggestField(state=sform.READONLY)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        if waffle.switch_is_active('remove_cities_from_vacancy_create_form'):
            self.fields.pop('cities', None)

        if not self.context.get('is_form_view'):
            return

        # Для этих полей нужно получить initial раньше остальных,
        # потому что все эти поля влияют на поле "доступы"
        access_related_fields = (
            'department',
            'hiring_manager',
            'instead_of',
        )
        self._cached_initial = {}
        for field_name in access_related_fields:
            field = self.fields.get(field_name)
            if field:
                self._get_initial_value(field_name, field)

    @cached_property
    def _department(self):
        # Для ручки формы это предустановленное подразделение,
        # для ручки создания – полученное от фронта
        if self.context.get('is_form_view'):
            return self._cached_initial.get('department')
        return self.cleaned_data.get('department')

    @cached_property
    def _membership(self):
        extra_access = []
        hiring_manager = self._cached_initial.get('hiring_manager')
        if hiring_manager:
            extra_access.append(hiring_manager)
        return IssueMembership(
            department=self._department,
            instead_of=self._cached_initial.get('instead_of'),
            extra_access=extra_access,
        )

    def get_head(self, initial):
        return self._membership.head.username if self._membership.head else None

    def get_access(self, initial):
        return [u.username for u in self._membership.access]

    def clean_locations(self, value):
        return value or None


class VacancyNewCreateForm(ConditionallyRequiredFieldsMixin, VacancyCreateBaseForm):
    """
    Форма создания новой вакансии
    """
    reason = sform.CharField()
    geography_international = sform.BooleanField(default=False, state=sform.REQUIRED)
    geography = sform.ModelChoiceField(
        queryset=(
            Geography.objects
            .alive()
            .exclude_roots()
            .filter(kind=GEOGRAPHY_KINDS.international)
        ),
        label_extractor=lambda x: x.name,
    )

    CONDITIONALLY_REQUIRED = {
        'geography': ('geography_international', True),
    }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['geography'].queryset = self.fields['geography'].queryset.order_by_tree()

    def clean(self):
        cleaned_data = super().clean()
        cleaned_data['type'] = choices.VACANCY_TYPES.new
        return cleaned_data


class VacancyReplacementCreateForm(ConditionallyRequiredFieldsMixin, VacancyCreateBaseForm):
    """
    Форма создания вакансии на замену
    """
    instead_of = UserSuggestField(state=sform.REQUIRED)
    replacement_reason = sform.ChoiceField(
        state=sform.REQUIRED,
        empty_label=EMPTY_LABEL,
        choices=choices.VACANCY_REPLACEMENT_REASONS,
    )
    quit_date = sform.DateField()
    contract_type = sform.ChoiceField(
        choices=choices.VACANCY_CONTRACT_TYPES,
        empty_label=EMPTY_LABEL,
    )
    replacement_department = DepartmentSuggestField()
    geography_international = sform.BooleanField(default=False)
    geography = sform.ModelChoiceField(
        queryset=(
            Geography.objects
            .alive()
            .exclude_roots()
            .filter(kind=GEOGRAPHY_KINDS.international)
        ),
        label_extractor=lambda x: x.name,
        state=sform.NORMAL,
    )

    CONDITIONALLY_REQUIRED = {
        'quit_date': ('replacement_reason', (
            choices.VACANCY_REPLACEMENT_REASONS.rotation,
            choices.VACANCY_REPLACEMENT_REASONS.dismissal,
        )),
        'contract_type': (
            'replacement_reason',
            choices.VACANCY_REPLACEMENT_REASONS.maternity_leave,
        ),
        'geography': ('geography_international', True),
    }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.fields['geography'].queryset = self.fields['geography'].queryset.order_by_tree()

    def clean(self):
        cleaned_data = super().clean()
        cleaned_data['type'] = choices.VACANCY_TYPES.replacement
        return cleaned_data


class VacancyInternshipCreateForm(VacancyCreateBaseForm):
    """
    Форма создания вакансии на стажировку
    """
    reason = sform.CharField(state=sform.REQUIRED)

    _exclude_fields = (
        'pro_level_min',
        'pro_level_max',
        'wage_system',
        'max_salary',
        'currency',
    )

    def clean(self):
        cleaned_data = super().clean()
        cleaned_data['type'] = choices.VACANCY_TYPES.internship
        cleaned_data['pro_level_min'] = choices.VACANCY_PRO_LEVELS.intern
        cleaned_data['pro_level_max'] = choices.VACANCY_PRO_LEVELS.intern
        return cleaned_data


class VacancyUpdateForm(ConditionallyRequiredFieldsMixin, BaseMetaForm, VacancyBaseForm):
    """
    Базовая форма редактирования вакансии
    """
    main_recruiter = UserSuggestField(state=sform.REQUIRED)
    recruiters = UserMultipleSuggestField()
    responsibles = HiringManagerLevelField(UserMultipleSuggestField)
    interviewers = ResponsibleLevelField(UserMultipleSuggestField)
    observers = ResponsibleLevelField(UserMultipleSuggestField)
    responsibilities_comment = ResponsibleLevelField(sform.CharField)
    additional_qualities_comment = ResponsibleLevelField(sform.CharField)
    recommendations_comment = ResponsibleLevelField(sform.CharField)
    kpi_comment = ResponsibleLevelField(sform.CharField)
    other_comment = ResponsibleLevelField(sform.CharField)
    is_published = ResponsibleLevelField(sform.BooleanField)
    publication_title = ResponsibleLevelField(sform.CharField, max_length=255)
    publication_content = ResponsibleLevelField(sform.CharField)

    value_stream = ResponsibleLevelField(
        sform.ModelChoiceField,
        queryset=ValueStream.objects.filter(
            is_active=True,
            oebs_product_id__isnull=False,
        ),
        label_extractor=lambda x: x.name,
    )
    work_mode = ResponsibleLevelField(
        WorkModeMultipleChoiceField,
    )

    # Поле, доступное для редактирования только руководителю подбора
    is_hidden = sform.BooleanField()

    # При создании эти поля обязательны, но при редактировании мы делаем их необязательными,
    # чтобы не пришлось их насильно добавлять при редактировании старых вакансий
    deadline = sform.DateField()
    cities = ResponsibleLevelField(CityMultipleSuggestField)

    # deleted from the creation form, but remained here for backwards compatibility
    offices = ResponsibleLevelField(OfficeMultipleChoiceField)
    priority = sform.ChoiceField(
        choices=choices.VACANCY_PRIORITIES,
        default=choices.VACANCY_PRIORITIES.normal,
    )
    skills = ResponsibleLevelField(SkillMultipleSuggestField)
    abc_services = ResponsibleLevelField(ServiceMultipleSuggestField)
    currency = sform.ModelChoiceField(
        queryset=Currency.active.order_by('id'),
        label_extractor=lambda x: x.code,
    )

    CONDITIONALLY_REQUIRED = {
        'publication_title': ('is_published', True),
    }

    ALWAYS_AVAILABLE_FIELDS = {
        'name',
        'skills',
        'interviewers',
        'observers',
        'responsibilities_comment',
        'additional_qualities_comment',
        'recommendations_comment',
        'kpi_comment',
        'other_comment',
        'publication_title',
        'publication_content',
        'is_published',
    }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.instance = self.context['instance']
        self.user = self.context['user']

        if self.user is None or not self.user.is_recruiting_manager:
            self.fields.pop('is_hidden')
        if not waffle.switch_is_active('enable_update_vacancy_abc_services'):
            self.fields.pop('abc_services', None)

    @property
    def _is_on_approval(self):
        return self.context['instance'].status == choices.VACANCY_STATUSES.on_approval

    def get_field_state(self, name):
        if self._is_on_approval and name not in self.ALWAYS_AVAILABLE_FIELDS:
            return sform.READONLY

        if is_role_field(self.fields[name]):
            return self.fields[name].get_state(self.instance, self.user)

        if not self.user.is_recruiter:
            return sform.READONLY

        if name == 'department' and self.department_edit_is_readonly():
            return sform.READONLY

        return super().get_field_state(name)

    def clean_main_recruiter(self, recruiter):
        validate_recruiters([recruiter])
        return recruiter

    def clean_recruiters(self, recruiters):
        validate_recruiters(recruiters)
        return recruiters

    def department_edit_is_readonly(self):
        return waffle.switch_is_active('disable_department_edit_in_favor_of_proposal')

    def meta_as_dict(self):
        return {
            'status': self.instance.status,
            'is_full_form': self.user.is_recruiter and not self._is_on_approval,
            'department_edit_moved_to_proposal': self.department_edit_is_readonly(),
        }


class VacancyReplacementUpdateForm(VacancyUpdateForm):
    """
    Форма редактирования вакансии на замену
    """
    instead_of = UserSuggestField(state=sform.REQUIRED)
    replacement_reason = sform.ChoiceField(
        state=sform.REQUIRED,
        empty_label=EMPTY_LABEL,
        choices=choices.VACANCY_REPLACEMENT_REASONS,
    )


class VacancyInternshipUpdateForm(VacancyUpdateForm):
    """
    Форма редактирования вакансии на стажировку
    """
    _exclude_fields = (
        'pro_level_min',
        'pro_level_max',
    )


class VacancyPoolUpdateForm(VacancyUpdateForm):
    """
    Форма редактирования пуловой вакансии
    """
    abc_services = ResponsibleLevelField(ServiceMultipleSuggestField)

    _exclude_fields = (
        'department',
    )


class VacancyApproveForm(sform.SForm):

    budget_position_id = PositiveIntegerField(
        max_value=PG_MAX_INTEGER_VALUE,
        state=sform.REQUIRED,
    )
    main_recruiter = UserSuggestField(state=sform.REQUIRED)
    recruiters = UserMultipleSuggestField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.instance = self.context.get('instance')

    def clean_budget_position_id(self, bp_id):
        exclude_ids = [self.instance.id] if self.instance else []
        return clean_budget_position_id(bp_id, exclude_ids)

    def clean_main_recruiter(self, recruiter):
        validate_recruiters([recruiter])
        return recruiter

    def clean_recruiters(self, recruiters):
        validate_recruiters(recruiters)
        return recruiters


class VacancyCommentForm(sform.SForm):

    comment = sform.CharField(state=sform.REQUIRED)


class StaffVacancyUpdateForm(sform.SForm):

    department = DepartmentSuggestField()
    value_stream = ValueStreamSuggestField(to_field_name='staff_id')
    geography = GeographySuggestField()
    startrek_approval_issue_key = sform.CharField()

    def clean(self):
        cleaned_data = super().clean()
        required_fields = (
            'department',
            'value_stream',
            'geography',
        )

        if not any(cleaned_data.get(f) for f in required_fields):
            raise ValidationError(
                message='Specify one of required fields: {}'.format(', '.join(required_fields)),
                code='missing_department_value_stream',
            )

        return cleaned_data


class VacancyAddToGroupForm(sform.SForm):

    vacancy_group = sform.ModelChoiceField(
        queryset=VacancyGroup.objects.filter(is_active=True),
        label_extractor=get_vacancy_group_label,
        state=sform.REQUIRED,
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        user = self.context.get('user')
        vacancy_group = self.fields['vacancy_group']
        vacancy_group.queryset = vacancy_group.queryset.filter(recruiters=user)


class VacancySuspendForm(VacancyCommentForm):

    need_close_applications = sform.BooleanField()
    application_resolution = sform.ChoiceField(
        choices=CLOSED_APPLICATION_RESOLUTIONS_MANAGEABLE,
        default=CLOSED_APPLICATION_RESOLUTIONS_MANAGEABLE.vacancy_closed,
    )


class VacancyCloseForm(VacancyCommentForm):

    application_resolution = sform.ChoiceField(
        choices=CLOSED_APPLICATION_RESOLUTIONS_MANAGEABLE,
        state=sform.REQUIRED,
        default=CLOSED_APPLICATION_RESOLUTIONS_MANAGEABLE.vacancy_closed,
    )


class VacancyListFilterForm(sform.SForm):

    default_getter = query_params_data_getter

    q = sform.CharField()
    role = sform.ChoiceField(
        choices=choices.VACANCY_FILTER_FORM_ROLES,
        empty_label=EMPTY_LABEL,
    )
    member = UserSuggestField()
    status = sform.ChoiceField(
        choices=choices.VACANCY_FILTER_FORM_STATUSES,
        default=choices.VACANCY_FILTER_FORM_STATUSES.open,
        empty_label=EMPTY_LABEL,
    )
    types = sform.MultipleChoiceField(
        choices=choices.VACANCY_TYPES,
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        user = self.context.get('user')
        if user:
            self.fields['member'].default = user.username
            if user.is_recruiter:
                self.fields['role'].choices = choices.VACANCY_ROLES

    def get_field_state(self, name):
        if name == 'member' and self.cleaned_data.get('role'):
            return sform.REQUIRED
        return super().get_field_state(name)


class StaffVacancyBPCreatedForm(sform.SForm):

    budget_position_id = sform.IntegerField(state=sform.REQUIRED)
