import waffle

from collections import (
    defaultdict,
    Counter,
    OrderedDict,
)

from django.db.models import Prefetch
from rest_framework import serializers

from intranet.femida.src.api.core.serializers import (
    AwareSerializerMixin,
    FemidaSerializer,
    WorkflowActionsField,
    IdNameSerializer,
    IdSlugNameSerializer,
)
from intranet.femida.src.api.professions.serializers import ProfessionLiteSerializer
from intranet.femida.src.api.skills.serializers import SkillSerializer
from intranet.femida.src.api.staff.serializers import DepartmentSerializer
from intranet.femida.src.api.submissions.serializers import (
    SubmissionFormSerializer,
    PublicationAsSubmissionFormSerializer,
)
from intranet.femida.src.api.users.serializers import UserSerializer
from intranet.femida.src.applications.helpers import active_applications_query
from intranet.femida.src.candidates.helpers import (
    get_candidates_with_last_active_application_for_vacancy,
)
from intranet.femida.src.interviews.choices import (
    INTERVIEW_TYPES,
    APPLICATION_STATUSES,
    CLOSED_APPLICATION_RESOLUTIONS_WITH_DEPRECATED,
    APPLICATION_SOURCES,
    APPLICATION_PROPOSAL_STATUSES,
)
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.startrek.utils import get_issue, StartrekError
from intranet.femida.src.vacancies import models
from intranet.femida.src.vacancies import choices
from intranet.femida.src.vacancies.workflow import VacancyWorkflow


class VacancyMembershipSerializer(FemidaSerializer):

    class Meta:
        model = models.VacancyMembership
        fields = ('id', 'vacancy', 'member', 'role')


class VacancySkillSerializer(FemidaSerializer):

    skill = SkillSerializer()

    class Meta:
        model = models.VacancySkill
        fields = ('id', 'vacancy', 'skill', 'is_required')


class VacancyHistorySerializer(FemidaSerializer):

    class Meta:
        model = models.VacancyHistory
        fields = '__all__'


class VacancyLiteSerializer(AwareSerializerMixin, FemidaSerializer):
    """
    Сериализатор, который нужно использовать для вакансий, вложенных в другие сущности
    """
    is_available = serializers.SerializerMethodField()
    department = DepartmentSerializer()

    def get_is_available(self, obj):
        if 'available_vacancy_ids' in self.root.context:
            return obj.id in self.root.context['available_vacancy_ids']

    class Meta:
        model = models.Vacancy
        fields = (
            'id',
            'name',
            'status',
            'department',
            'is_hidden',
            'is_available',
        )


class VacancyListItemSerializer(AwareSerializerMixin, FemidaSerializer):

    head = UserSerializer()
    hiring_manager = UserSerializer()
    main_recruiter = UserSerializer()
    recruiters = UserSerializer(many=True, source='researchers')
    department = DepartmentSerializer()
    value_stream = IdSlugNameSerializer()
    geography_international = serializers.ReadOnlyField()
    geography = IdNameSerializer()
    is_open = serializers.SerializerMethodField()
    cities = IdNameSerializer(many=True)

    def get_is_open(self, obj):
        return obj.status in choices.OPEN_VACANCY_STATUSES

    class Meta:
        model = models.Vacancy
        fields = (
            'id',
            'name',
            'status',
            'startrek_key',
            'modified',
            'head',
            'hiring_manager',
            'main_recruiter',
            'recruiters',
            'department',
            'value_stream',
            'geography_international',
            'geography',
            'is_open',
            'is_hidden',
            'cities',
        )
        select_related_map = {
            'department': ('department',),
            'value_stream': ('value_stream',),
            'geography': ('geography',),
        }
        prefetch_related_map = {
            'head': ('memberships__member',),
            'hiring_manager': ('memberships__member',),
            'main_recruiter': ('memberships__member',),
            'recruiters': ('memberships__member',),
            'cities': ('cities',),
        }


class VacancyDetailSerializer(AwareSerializerMixin, FemidaSerializer):

    profession = ProfessionLiteSerializer()
    vacancy_skills = VacancySkillSerializer(many=True)
    memberships = VacancyMembershipSerializer(many=True)
    hiring_manager = UserSerializer()
    head = UserSerializer()
    main_recruiter = UserSerializer()
    recruiters = UserSerializer(many=True, source='researchers')
    responsibles = UserSerializer(many=True)
    interviewers = UserSerializer(many=True)
    observers = UserSerializer(many=True)
    instead_of = UserSerializer()
    created_by = UserSerializer()
    department = DepartmentSerializer()
    value_stream = IdSlugNameSerializer()
    work_mode = IdSlugNameSerializer(many=True)
    geography_international = serializers.ReadOnlyField()
    geography = IdNameSerializer()
    actions = WorkflowActionsField(workflow_class=VacancyWorkflow)
    active_applications = serializers.SerializerMethodField()
    candidates_with_last_active_application = serializers.SerializerMethodField()
    pro_level_min_display = serializers.ReadOnlyField(source='get_pro_level_min_display')
    pro_level_max_display = serializers.ReadOnlyField(source='get_pro_level_max_display')
    pro_level_min_verbose = serializers.ReadOnlyField(source='get_pro_level_min_display')
    pro_level_max_verbose = serializers.ReadOnlyField(source='get_pro_level_max_display')
    submission_forms = serializers.SerializerMethodField()
    publications = PublicationAsSubmissionFormSerializer(many=True)
    cities = IdNameSerializer(many=True)
    offices = IdNameSerializer(many=True)
    locations = IdNameSerializer(many=True)
    skills = IdNameSerializer(many=True)
    abc_services = IdNameSerializer(many=True)

    def get_active_applications(self, obj):
        return (
            obj.applications
            .filter(active_applications_query)
            .values_list('id', flat=True)
        )

    def get_candidates_with_last_active_application(self, obj):
        return (
            get_candidates_with_last_active_application_for_vacancy(obj)
            .values('id', 'first_name', 'last_name')
        )

    def get_submission_forms(self, obj):
        publications = PublicationAsSubmissionFormSerializer(obj.publications.all(), many=True).data
        if not waffle.switch_is_active('disable_submission_forms_in_serializers'):
            publications += SubmissionFormSerializer(obj.submission_forms.all(), many=True).data
        return publications

    class Meta:
        model = models.Vacancy
        exclude = (
            'publication_content',
            'formatted_publication_content',
        )
        select_related_map = {
            'profession': ('profession__professional_sphere',),
            'instead_of': ('instead_of',),
            'created_by': ('created_by',),
            'department': ('department',),
            'value_stream': ('value_stream',),
            'geography': ('geography',),
        }
        prefetch_related_map = {
            'vacancy_skills': ('vacancy_skills__skill',),
            'skills': ('skills',),
            'cities': ('cities',),
            'memberships': ('memberships__member',),
            'hiring_manager': ('memberships__member',),
            'head': ('memberships__member',),
            'main_recruiter': ('memberships__member',),
            'recruiters': ('memberships__member',),
            'responsibles': ('memberships__member',),
            'interviewers': ('memberships__member',),
            'observers': ('memberships__member',),
            'submission_forms': ('submission_forms', 'publications'),
            'publications': ('publications',),
        }


class VacancyStatsSerializer(AwareSerializerMixin, FemidaSerializer):

    history = VacancyHistorySerializer(many=True, source='vacancy_history')
    applications_statuses = serializers.SerializerMethodField()
    applications_sources = serializers.SerializerMethodField()
    interviews = serializers.SerializerMethodField()
    candidates_interviews = serializers.SerializerMethodField()
    offers = serializers.SerializerMethodField()
    candidates_offers = serializers.SerializerMethodField()
    proposals = serializers.SerializerMethodField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.candidate_ids = {a.candidate_id for a in self.instance.applications.all()}

    def get_applications_statuses(self, obj):
        data = defaultdict(Counter)
        closed_resolutions = (i for i, _ in CLOSED_APPLICATION_RESOLUTIONS_WITH_DEPRECATED)
        data[APPLICATION_STATUSES.closed] = OrderedDict.fromkeys(closed_resolutions, 0)
        data[APPLICATION_STATUSES.closed]['total'] = 0
        data[APPLICATION_STATUSES.closed]['no_resolution'] = 0

        for application in obj.applications.all():
            data[application.status]['total'] += 1
            resolution = application.resolution or 'no_resolution'
            data[application.status][resolution] += 1
        return data

    def get_applications_sources(self, obj):
        data = Counter()
        for application in obj.applications.all():
            data[application.source] += 1

        return data

    def get_interviews(self, obj):
        data = {k: 0 for k in INTERVIEW_TYPES._db_values}
        data['total'] = 0
        for application in obj.applications.all():
            interviews = self.root.context['interviews_by_application'][application.id]
            for interview in interviews:
                data['total'] += 1
                data[interview['type']] += 1
        return data

    def get_candidates_interviews(self, obj):
        data = {k: 0 for k in INTERVIEW_TYPES._db_values}
        data['total'] = 0
        for candidate_id in self.candidate_ids:
            interviews = self.root.context['interviews_by_candidate'][candidate_id]
            for interview in interviews:
                data['total'] += 1
                data[interview['type']] += 1
        return data

    def get_offers(self, obj):
        data = {
            'total': 0,
        }
        for application in obj.applications.all():
            offers = self.root.context['offers_by_application'][application.id]
            data['total'] += len(offers)
        return data

    def get_candidates_offers(self, obj):
        data = {
            'total': 0,
        }
        for candidate_id in self.candidate_ids:
            offers = self.root.context['offers_by_candidate'][candidate_id]
            data['total'] += len(offers)
        return data

    def get_proposals(self, obj):
        defaults = dict.fromkeys(
            APPLICATION_PROPOSAL_STATUSES._db_values
            | {'total', 'regular_interviews'},
            0,
        )
        data = Counter(defaults)
        proposals = (a for a in obj.applications.all() if a.source == APPLICATION_SOURCES.proposal)
        for application in proposals:
            data['total'] += 1
            data[application.proposal_status] += 1
            data['regular_interviews'] += any(
                i['type'] == INTERVIEW_TYPES.regular
                for i in self.root.context['interviews_by_application'][application.id]
            )
        return data

    class Meta:
        model = models.Vacancy
        fields = (
            'id',
            'name',
            'status',
            'resolution',
            'history',
            'applications_statuses',
            'applications_sources',
            'interviews',
            'offers',
            'candidates_interviews',
            'candidates_offers',
            'proposals',
        )

        prefetch_related_map = {
            'id': ('applications',),
            'history': (
                Prefetch(
                    lookup='vacancy_history',
                    queryset=models.VacancyHistory.objects.order_by('changed_at'),
                ),
            ),
        }


# FORM SERIALIZERS


class VacancySkillFormSerializer(serializers.ModelSerializer):

    class Meta:
        model = models.VacancySkill
        fields = '__all__'


class VacancyFormSerializer(serializers.ModelSerializer):

    vacancy_skills = VacancySkillFormSerializer(many=True)
    instead_of = serializers.CharField(source='instead_of.username', default=None)
    hiring_manager = serializers.CharField(source='hiring_manager.username', default=None)
    head = serializers.CharField(source='head.username', default=None)
    main_recruiter = serializers.CharField(source='main_recruiter.username', default=None)
    recruiters = serializers.SerializerMethodField()
    responsibles = serializers.SerializerMethodField()
    interviewers = serializers.SerializerMethodField()
    observers = serializers.SerializerMethodField()
    abc_services = serializers.SerializerMethodField()
    locations = serializers.SerializerMethodField()

    def get_recruiters(self, obj):
        return [u.username for u in obj.researchers]

    def get_responsibles(self, obj):
        return [u.username for u in obj.responsibles]

    def get_interviewers(self, obj):
        return [u.username for u in obj.interviewers]

    def get_observers(self, obj):
        return [u.username for u in obj.observers]

    def get_abc_services(self, obj):
        return [s.id for s in obj.abc_services.all()]

    def get_locations(self, obj):
        return [location.geo_id for location in obj.locations.all()]

    class Meta:
        model = models.Vacancy
        fields = '__all__'


class VacancyApproveFormSerializer(FemidaSerializer):

    budget_position_id = serializers.SerializerMethodField()

    def get_budget_position_id(self, obj):
        try:
            issue = get_issue(obj.startrek_key)
        except StartrekError:
            pass
        else:
            return issue.bpNumber

    class Meta:
        model = models.Vacancy
        fields = (
            'budget_position_id',
        )


class StaffOfferForVacancySerializer(FemidaSerializer):

    candidate = serializers.SerializerMethodField()

    def get_candidate(self, obj):
        return {
            'id': obj.pop('candidate_id'),
            'first_name': obj.pop('first_name'),
            'middle_name': obj.pop('middle_name'),
            'last_name': obj.pop('last_name'),
        }

    class Meta:
        model = Offer
        fields = (
            'id',
            'newhire_id',
            'username',
            'candidate',
            'application_id',
            'startrek_salary_key',
        )


class StaffVacancySerializer(FemidaSerializer):

    offer = StaffOfferForVacancySerializer()
    profession_staff_id = serializers.CharField()
    department_url = serializers.CharField()
    department_group_id = serializers.CharField()
    access = serializers.ListField()

    class Meta:
        model = models.Vacancy
        fields = (
            'id',
            'name',
            'status',
            'budget_position_id',
            'startrek_key',
            'created',
            'modified',
            'offer',
            'access',
            'is_published',
            'is_hidden',
            'profession_staff_id',
            'department_url',
            'department_group_id',
        )
