import logging

from collections import defaultdict

from django.db.models import Q, OuterRef, F
from rest_framework import status
from rest_framework.exceptions import ParseError as BadRequest
from rest_framework.response import Response

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.api.core.views import (
    BaseView,
    WorkflowView,
    WorkflowViewMixin,
    BaseFormView,
    BaseFormViewMixin,
    ValidatorSwitcherViewMixin,
    InstanceFormViewMixin,
    QueryParamValidatorSwitcherViewMixin,
    unfold_query_params,
)
from intranet.femida.src.api.core.pagination import CursorPagination
from intranet.femida.src.api.core.permissions import ServicePermission
from intranet.femida.src.api.hire_orders.permissions import ForbiddenForAutohire
from intranet.femida.src.api.staff.permissions import StaffPermission
from intranet.femida.src.api.tracker.views import TrackerTriggerBaseView
from intranet.femida.src.interviews.models import Interview, Application
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.startrek.utils import StartrekError
from intranet.femida.src.vacancies import choices
from intranet.femida.src.vacancies.tasks import (
    vacancy_close_by_issue_task,
    vacancy_change_type_by_issue_task,
    vacancy_approve_bp_by_issue_task,
    vacancy_approve_by_issue_task,
    vacancy_bp_created_task,
)
from intranet.femida.src.vacancies.controllers import update_or_create_vacancy
from intranet.femida.src.vacancies.models import Vacancy, VacancyMembership
from intranet.femida.src.vacancies.workflow import VacancyWorkflow
from intranet.femida.src.offers.choices import OFFER_STATUSES
from intranet.femida.src.core.db import RowToDict, ArraySubquery
from . import forms, serializers


logger = logging.getLogger(__name__)


class VacancyCreateMixin(QueryParamValidatorSwitcherViewMixin):

    validator_classes_map = {
        choices.VACANCY_TYPES.new: forms.VacancyNewCreateForm,
        choices.VACANCY_TYPES.replacement: forms.VacancyReplacementCreateForm,
        choices.VACANCY_TYPES.internship: forms.VacancyInternshipCreateForm,
    }


class VacancyListCreateView(VacancyCreateMixin, BaseView):

    model_class = Vacancy
    list_item_serializer_class = serializers.VacancyListItemSerializer
    detail_serializer_class = serializers.VacancyDetailSerializer

    # Форма по умолчанию при создании вакансии
    validator_class = forms.VacancyNewCreateForm

    def get_key(self):
        return self.request.data.get('type', choices.VACANCY_TYPES.new)

    def get_search_query(self, query_term):
        query = Q(name__icontains=query_term) | Q(startrek_key__icontains=query_term)
        try:
            query |= Q(id=int(query_term))
        except ValueError:
            pass
        return query

    def filter_queryset(self, queryset):
        params = unfold_query_params(
            query_params=self.request.query_params,
            list_fields=('types',),
        )
        filter_form = forms.VacancyListFilterForm(
            data=params,
            context={
                'user': self.request.user,
            },
        )
        self.validate(filter_form)

        filter_params = filter_form.cleaned_data
        if filter_params['q']:
            queryset = queryset.filter(self.get_search_query(filter_params['q']))

        if filter_params['member']:
            memberships_qs = (
                VacancyMembership.unsafe
                .filter(member=filter_params['member'])
            )
            if filter_params['role']:
                memberships_qs = memberships_qs.filter(role=filter_params['role'])

            queryset = queryset.filter(id__in=memberships_qs.values('vacancy'))

        # TODO: выпилить, если откажемся от групп вакансий
        if 'group_id' in params:
            queryset = queryset.filter(groups=params['group_id'])

        if filter_params['status']:
            if filter_params['status'] == choices.VACANCY_FILTER_FORM_STATUSES.open:
                queryset = queryset.filter(status__in=choices.OPEN_VACANCY_STATUSES._db_values)
            else:
                queryset = queryset.filter(status=filter_params['status'])

        if filter_params['types']:
            queryset = queryset.filter(type__in=filter_params['types'])

        return queryset.order_by('-modified')

    def get(self, request, *args, **kwargs):
        """
        Список вакансий
        """
        return self.list(request, *args, **kwargs)

    @action_logged('vacancy_create')
    def post(self, request, *args, **kwargs):
        """
        Создание вакансии
        """
        try:
            return self.create(request, *args, **kwargs)
        except StartrekError as exc:
            raise BadRequest(exc.code)

    def perform_create(self, data):
        return update_or_create_vacancy(
            data=data,
            initiator=self.request.user,
        )


class VacancyDetailView(ValidatorSwitcherViewMixin, InstanceFormViewMixin,
                        WorkflowViewMixin, BaseView):

    model_class = Vacancy
    detail_serializer_class = serializers.VacancyDetailSerializer
    form_serializer_class = serializers.VacancyFormSerializer
    workflow_class = VacancyWorkflow

    validator_class = forms.VacancyUpdateForm
    validator_classes_map = {
        choices.VACANCY_TYPES.new: forms.VacancyUpdateForm,
        choices.VACANCY_TYPES.replacement: forms.VacancyReplacementUpdateForm,
        choices.VACANCY_TYPES.internship: forms.VacancyInternshipUpdateForm,
        choices.VACANCY_TYPES.pool: forms.VacancyPoolUpdateForm,
    }

    def get_validator_context(self):
        return {
            'instance': self.get_object(),
        }

    def get_permissions(self):
        permissions = super().get_permissions()
        if not getattr(self, 'is_allowed_for_autohire', False):
            permissions.append(ForbiddenForAutohire('vacancy_update'))
        return permissions

    def get(self, request, *args, **kwargs):
        """
        Одна вакансия
        """
        self.is_allowed_for_autohire = True
        return self.retrieve(request, *args, **kwargs)

    @action_logged('vacancy_update')
    def put(self, request, *args, **kwargs):
        """
        Редактирование вакансии
        """
        return self.perform_action('update')


class VacancyStatsView(BaseView):

    model_class = Vacancy
    detail_serializer_class = serializers.VacancyStatsSerializer

    def get_detail_serializer_context(self):
        candidates_qs = (
            Application.unsafe
            .filter(vacancy_id=self.kwargs['pk'])
            .values('candidate_id')
        )
        interviews = list(
            Interview.unsafe.alive()
            .filter(candidate__in=candidates_qs)
            .values(
                'id',
                'type',
                'candidate_id',
                'application_id',
            )
        )
        interviews_by_candidate = defaultdict(list)
        interviews_by_application = defaultdict(list)
        for interview in interviews:
            interviews_by_candidate[interview['candidate_id']].append(interview)
            interviews_by_application[interview['application_id']].append(interview)

        offers = list(
            Offer.unsafe
            .filter(candidate__in=candidates_qs)
            .values(
                'id',
                'application_id',
                'candidate_id',
            )
        )
        offers_by_candidate = defaultdict(list)
        offers_by_application = defaultdict(list)
        for offer in offers:
            offers_by_candidate[offer['candidate_id']].append(offer)
            offers_by_application[offer['application_id']].append(offer)

        return {
            'interviews_by_candidate': interviews_by_candidate,
            'interviews_by_application': interviews_by_application,
            'offers_by_candidate': offers_by_candidate,
            'offers_by_application': offers_by_application,
        }

    def get(self, request, *args, **kwargs):
        """
        Статистика по вакансии
        """
        return self.retrieve(request, *args, **kwargs)


class VacancyListFilterFormView(BaseFormView):

    model_class = Vacancy
    validator_class = forms.VacancyListFilterForm

    def get_initial_data(self):
        return unfold_query_params(
            query_params=self.request.query_params,
            list_fields=('types',),
        )


class VacancyCreateFormView(VacancyCreateMixin, BaseFormView):

    model_class = Vacancy
    form_serializer_class = serializers.VacancyFormSerializer

    # Форма по умолчанию в ручке формы создания
    validator_class = forms.VacancyChooseTypeForm

    def get_validator_context(self):
        return {
            'is_form_view': True,
        }

    def get_initial_data(self):
        return unfold_query_params(
            query_params=self.request.query_params,
            list_fields=(
                'abc_services',
                'cities',
                'followers',
                'skills',
            ),
        )


class VacancyWorkflowView(WorkflowView):

    model_class = Vacancy
    detail_serializer_class = serializers.VacancyDetailSerializer
    workflow_class = VacancyWorkflow
    permission_classes = [ForbiddenForAutohire]

    @property
    def actionlog_name(self):
        return 'vacancy_%s' % self.action_name

    def get_additional_response_data(self):
        return self.action.extra_data


class VacancyApproveView(InstanceFormViewMixin, VacancyWorkflowView):

    action_name = 'approve'
    validator_class = forms.VacancyApproveForm
    form_serializer_class = serializers.VacancyApproveFormSerializer

    def get_validator_context(self):
        return {
            'instance': self.get_object(),
        }


class VacancyAddToGroupView(BaseFormViewMixin, VacancyWorkflowView):

    action_name = 'add_to_group'
    validator_class = forms.VacancyAddToGroupForm


class VacancyCloseView(BaseFormViewMixin, VacancyWorkflowView):

    action_name = 'close'

    def get_validator_class(self):
        if self.get_object().status == choices.VACANCY_STATUSES.on_approval:
            return forms.VacancyCommentForm
        return forms.VacancyCloseForm


class VacancySuspendView(BaseFormViewMixin, VacancyWorkflowView):

    action_name = 'suspend'
    validator_class = forms.VacancySuspendForm


class VacancyResumeView(BaseFormViewMixin, VacancyWorkflowView):

    action_name = 'resume'
    validator_class = forms.VacancyCommentForm


class StaffVacancyListView(BaseView):

    model_class = Vacancy
    list_item_serializer_class = serializers.StaffVacancySerializer
    permission_classes = (StaffPermission,)
    pagination_class = CursorPagination

    def get_queryset(self):
        return (
            self.model_class.unsafe
            .exclude(
                type=choices.VACANCY_TYPES.pool,
            )
            .annotate(
                offer=RowToDict(
                    Offer.unsafe
                    .filter(
                        vacancy=OuterRef('id'),
                        newhire_id__isnull=False,
                        status__in=(
                            OFFER_STATUSES.sent,
                            OFFER_STATUSES.accepted,
                        ),
                    )
                    .values(
                        'id',
                        'newhire_id',
                        'username',
                        'candidate_id',
                        'application_id',
                        'startrek_salary_key',
                        first_name=F('profile__first_name'),
                        middle_name=F('profile__middle_name'),
                        last_name=F('profile__last_name'),
                    )[:1]
                ),
                access=ArraySubquery(
                    VacancyMembership.unsafe
                    .filter(
                        vacancy=OuterRef('id'),
                    )
                    .values_list('member__username', flat=True)
                ),
            )
            .values(
                'id',
                'name',
                'status',
                'budget_position_id',
                'startrek_key',
                'created',
                'modified',
                'offer',
                'access',
                'is_published',
                'is_hidden',
                profession_staff_id=F('profession__staff_id'),
                department_url=F('department__url'),
                department_group_id=F('department__group_id'),
            )
        )

    def get(self, *args, **kwargs):
        return self.list(*args, **kwargs)


class StaffVacancyUpdateView(VacancyWorkflowView):

    action_name = 'update_by_staff'
    permission_classes = (
        ServicePermission('permissions.can_use_api_for_staff'),
        StaffPermission,
    )
    validator_class = forms.StaffVacancyUpdateForm

    def get_queryset(self):
        return self.model_class.unsafe.all()

    def post(self, request, *args, **kwargs):
        vacancy = self.get_object()

        if vacancy.status == choices.VACANCY_STATUSES.closed:
            logger.info('Action skipped because vacancy is closed')
            return Response({}, status=status.HTTP_304_NOT_MODIFIED)
        elif self._offer_is_on_approval(vacancy):
            logger.info('Action skipped because offer is on approval')
            return Response({}, status=status.HTTP_304_NOT_MODIFIED)

        return super().post(request, *args, **kwargs)

    @staticmethod
    def _offer_is_on_approval(vacancy):
        offer_approval_states = (OFFER_STATUSES.on_approval, OFFER_STATUSES.on_rotation_approval)
        offer = vacancy.offers.alive().first()

        return (
            vacancy.status == choices.VACANCY_STATUSES.offer_processing
            and offer
            and offer.status in offer_approval_states
        )


class StaffVacancyBPCreatedView(BaseFormView, BaseView):
    """
    Ручка для стаффа, дергается после создания бюджетной позиции
    """
    model_class = Vacancy
    permission_classes = (StaffPermission,)
    validator_class = forms.StaffVacancyBPCreatedForm

    def get_queryset(self):
        return self.model_class.unsafe.all()

    def post(self, request, *args, **kwargs):
        instance = self.get_object()

        validator = self.get_validator_object(self.request.data)
        self.validate(validator)
        budget_position_id = validator.cleaned_data['budget_position_id']
        vacancy_bp_created_task.delay(instance.id, budget_position_id)
        return Response(status=status.HTTP_200_OK)


class StaffVacancyCloseView(VacancyWorkflowView):

    action_name = 'close_by_bp_registry'
    permission_classes = (StaffPermission,)
    validator_class = forms.VacancyCommentForm

    def get_queryset(self):
        return self.model_class.unsafe.all()


class VacancyEditByIssue(TrackerTriggerBaseView):

    model_class = Vacancy
    issue_key_field = 'startrek_key'
    valid_statuses = (
        choices.VACANCY_STATUSES.on_approval,
    )


class VacancyCloseByIssueView(VacancyEditByIssue):
    """
    Ручка для Трекера на событие "тикет закрыт"
    """
    task = vacancy_close_by_issue_task
    valid_statuses = (
        choices.VACANCY_STATUSES.in_progress,
        choices.VACANCY_STATUSES.suspended,
        choices.VACANCY_STATUSES.on_approval,
    )


class VacancyChangeTypeByIssueView(VacancyEditByIssue):
    """
    Ручка для Трекера на событие "изменилось поле стажировка"
    """
    task = vacancy_change_type_by_issue_task


class VacancyApproveBPByIssueView(VacancyEditByIssue):
    """
    Ручка для Трекера на событие "Согласование БП по вакансии завершено"
    """
    task = vacancy_approve_bp_by_issue_task


class VacancyApproveByIssueView(VacancyEditByIssue):
    """
    Ручка для Трекера на событие "Согласование вакансии завершено"
    """
    task = vacancy_approve_by_issue_task
