from django.forms.fields import BooleanField
from rest_framework.exceptions import PermissionDenied

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.api.core.views import (
    BaseView,
    WorkflowView,
    InstanceFormViewMixin,
    BaseFormViewMixin,
)
from intranet.femida.src.vacancies.controllers import update_or_create_vacancy_group
from intranet.femida.src.vacancies.models import VacancyGroup
from intranet.femida.src.vacancies.vacancy_groups.workflow import VacancyGroupWorkflow
from . import forms
from . import serializers


class VacancyGroupListCreateView(BaseFormViewMixin, BaseView):

    model_class = VacancyGroup
    validator_class = forms.VacancyGroupForm
    detail_serializer_class = serializers.VacancyGroupDetailSerializer
    list_item_serializer_class = serializers.VacancyGroupDetailSerializer

    def get_queryset(self):
        return (
            self.model_class.objects
            .select_related('created_by')
            .prefetch_related('vacancies', 'recruiters')
        )

    def filter_queryset(self, queryset):
        params = self.request.query_params

        if 'is_active' in params:
            is_active = BooleanField().to_python(params['is_active'])
            queryset = queryset.filter(is_active=is_active)

        if 'recruiter' in params:
            queryset = queryset.filter(recruiters__username=params['recruiter'])

        return queryset.order_by('-modified')

    def get(self, request, *args, **kwargs):
        """
        Список с группами вакансий.
        """
        return self.list(request, *args, **kwargs)

    @action_logged('vacancy_group_create')
    def post(self, request, *args, **kwargs):
        """
        Создание группы вакансий
        """
        return self.create(request, *args, **kwargs)

    def perform_create(self, data):
        return update_or_create_vacancy_group(data, self.request.user)


class VacancyGroupDetailView(InstanceFormViewMixin, BaseView):

    model_class = VacancyGroup
    detail_serializer_class = serializers.VacancyGroupDetailSerializer
    form_serializer_class = serializers.VacancyGroupFormSerializer
    validator_class = forms.VacancyGroupForm

    def get(self, request, *args, **kwargs):
        """
        Одна группа вакансий
        """
        return self.retrieve(request, *args, **kwargs)

    @action_logged('vacancy_group_update')
    def put(self, request, *args, **kwargs):
        """
        Редактирование группы вакансий
        """
        return self.update(request, *args, **kwargs)

    def perform_update(self, data, instance):
        workflow = VacancyGroupWorkflow(instance, self.request.user)
        update_action = workflow.get_action('update')
        if not update_action.is_available():
            raise PermissionDenied('workflow_error')

        return update_action.perform(**data)


class VacancyGroupWorkflowView(WorkflowView):

    model_class = VacancyGroup
    detail_serializer_class = serializers.VacancyGroupDetailSerializer
    workflow_class = VacancyGroupWorkflow

    @property
    def actionlog_name(self):
        return 'vacancy_group_%s' % self.action_name


class VacancyGroupActivateView(VacancyGroupWorkflowView):

    action_name = 'activate'


class VacancyGroupDeactivateView(VacancyGroupWorkflowView):

    action_name = 'deactivate'
