from typing import List

from django.utils import timezone

from intranet.femida.src.applications.exceptions import (
    InactiveVacancyError,
    ApplicationAlreadyExists,
)
from intranet.femida.src.applications.helpers import active_applications_query
from intranet.femida.src.applications.models import Application
from intranet.femida.src.applications.tasks import notify_vacancy_applications_closed_task
from intranet.femida.src.core.controllers import update_instance
from intranet.femida.src.core.signals import post_update
from intranet.femida.src.interviews.choices import (
    APPLICATION_RESOLUTIONS,
    APPLICATION_STATUSES,
    APPLICATION_CLOSE_REASONS,
)
from intranet.femida.src.utils.itertools import get_chunks
from intranet.femida.src.vacancies.choices import ACTIVE_VACANCY_STATUSES


def update_or_create_application(data, initiator=None, instance=None):

    if instance is not None:
        # Изменение всех видов статусов -- часть workflow. Должны явно выполнять в нем
        data.pop('status', None)
        data.pop('resolution', None)
        data.pop('proposal_status', None)
        update_instance(instance, data)
    else:
        data['created_by'] = initiator
        instance = Application.objects.create(**data)
    return instance


def create_application_gracefully(data, initiator=None):
    candidate = data['candidate']
    vacancy = data['vacancy']
    if vacancy.status not in ACTIVE_VACANCY_STATUSES:
        raise InactiveVacancyError

    active_applications = candidate.applications.filter(active_applications_query, vacancy=vacancy)
    if active_applications.exists():
        raise ApplicationAlreadyExists

    return update_or_create_application(data, initiator)


def bulk_close_applications_by_id(application_ids: List[int], resolution, notify=False,
                                  reason=None, initiator=None):
    modified = timezone.now()
    applications = Application.unsafe.filter(id__in=application_ids)

    # Закрываем прет-ва
    applications.update(
        status=APPLICATION_STATUSES.closed,
        resolution=resolution,
        modified=modified,
    )
    post_update.send(
        sender=Application,
        queryset=applications,
    )

    if notify:
        # Отправляем комменты и нотификации
        # Прет-в может быть неприлично много, поэтому отправляем по 100 за раз
        for chunk in get_chunks(application_ids, 100):
            notify_vacancy_applications_closed_task.delay(
                application_ids=chunk,
                reason=reason,
                initiator_id=initiator.id if initiator else None,
            )


def bulk_close_applications(queryset, resolution, reason, initiator=None):
    application_ids = list(queryset.values_list('id', flat=True))
    bulk_close_applications_by_id(
        application_ids=application_ids,
        resolution=resolution,
        notify=True,
        reason=reason,
        initiator=initiator,
    )


def close_vacancy_applications(vacancy, resolution=APPLICATION_RESOLUTIONS.vacancy_closed,
                               initiator=None):
    """
    Закрывает активные прет-ва на вакансии
    """
    queryset = vacancy.applications.filter(active_applications_query)
    bulk_close_applications(
        queryset=queryset,
        resolution=resolution,
        reason=APPLICATION_CLOSE_REASONS.vacancy_closed,
        initiator=initiator,
    )
