from django.db.models import Case, When, Value, BooleanField

from intranet.femida.src.applications.helpers import application_has_interview_qs
from intranet.femida.src.interviews.choices import (
    APPLICATION_DASHBOARD_STAGES as STAGES,
    APPLICATION_STATUSES,
    APPLICATION_PROPOSAL_STATUSES,
    APPLICATION_RESOLUTIONS,
    INTERVIEW_TYPES,
    CLOSED_APPLICATION_RESOLUTIONS,
)


class StageFilter:

    def __init__(self, **kwargs):
        self._data = kwargs
        self._data.setdefault('status', APPLICATION_STATUSES.in_progress)

    def filter(self, qs):
        for param, value in self._data.items():
            annotate = getattr(self, f'_annotate_{param}', lambda x: x)
            qs = annotate(qs)
        return qs.filter(**self._data)

    @staticmethod
    def _annotate_has_interview(qs):
        return qs.annotate(has_interview=application_has_interview_qs())

    @staticmethod
    def _annotate_has_screening_interview(qs):
        annotation = application_has_interview_qs(type=INTERVIEW_TYPES.screening)
        return qs.annotate(has_screening_interview=annotation)

    @staticmethod
    def _annotate_has_regular_interview(qs):
        annotation = application_has_interview_qs(type=INTERVIEW_TYPES.regular)
        return qs.annotate(has_regular_interview=annotation)

    @staticmethod
    def _annotate_has_final_interview(qs):
        annotation = application_has_interview_qs(type=INTERVIEW_TYPES.final)
        return qs.annotate(has_final_interview=annotation)

    @staticmethod
    def _annotate_has_offer(qs):
        return qs.annotate(has_offer=Case(
            When(resolution=APPLICATION_RESOLUTIONS.offer_agreement, then=Value(True)),
            default=Value(False),
            output_field=BooleanField(),
        ))


_STAGE_TO_FILTER = {
    STAGES.draft: StageFilter(status=APPLICATION_STATUSES.draft),
    STAGES.new: StageFilter(
        proposal_status=APPLICATION_PROPOSAL_STATUSES.undefined,
        has_interview=False,
        has_offer=False,
    ),
    STAGES.team_is_interested: StageFilter(
        proposal_status=APPLICATION_PROPOSAL_STATUSES.accepted,
        has_interview=False,
        has_offer=False,
    ),
    STAGES.invited_to_preliminary_interview: StageFilter(
        has_screening_interview=True,
        has_regular_interview=False,
        has_final_interview=False,
        has_offer=False,
    ),
    STAGES.invited_to_onsite_interview: StageFilter(
        has_regular_interview=True,
        has_final_interview=False,
        has_offer=False,
    ),
    STAGES.invited_to_final_interview: StageFilter(
        has_final_interview=True,
        has_offer=False,
    ),
    STAGES.offer_agreement: StageFilter(has_offer=True),
    STAGES.closed: StageFilter(status=APPLICATION_STATUSES.closed),

    # Стадии для закрытых прет-ов
    **{
        resolution: StageFilter(status=APPLICATION_STATUSES.closed, resolution=resolution)
        for resolution, _ in CLOSED_APPLICATION_RESOLUTIONS
    },
}


def filter_applications_by_dashboard_stage(qs, stage):
    if stage in _STAGE_TO_FILTER:
        rule = _STAGE_TO_FILTER[stage]
        return rule.filter(qs)
    return qs.none()
