from django.db.models import Q, OuterRef, Count, F, Exists, Case, When, CharField
from django.db.models.functions import Concat
from django.contrib.auth import get_user_model

from intranet.femida.src.core.db import RowsToList, RowToDict, Value
from intranet.femida.src.core.db.helpers import get_count_subquery
from intranet.femida.src.candidates.models import Consideration
from intranet.femida.src.interviews.choices import (
    APPLICATION_STATUSES,
    INTERVIEW_TYPES,
    APPLICATION_HIRING_STAGES,
    APPLICATION_PROPOSAL_STATUSES,
)
from intranet.femida.src.interviews.models import Interview
from intranet.femida.src.communications.models import Message
from intranet.femida.src.communications.choices import COMMENT_MESSAGE_TYPES


User = get_user_model()
archived_applications_query = Q(status=APPLICATION_STATUSES.closed)
active_applications_query = ~archived_applications_query


def _has_interview_qs(**filter_params):
    return Exists(
        Interview.unsafe
        .alive()
        .filter(**filter_params)
        .values('id')
    )


def _candidate_has_interview_qs(**filter_params):
    return _has_interview_qs(consideration=OuterRef('consideration'), **filter_params)


def application_has_interview_qs(**filter_params):
    return _has_interview_qs(application=OuterRef('id'), **filter_params)


def _annotate_applications_qs_with_interview_flags(qs):
    return (
        qs
        .annotate(
            has_regular_interview=application_has_interview_qs(type=INTERVIEW_TYPES.regular),
            has_final_interview=application_has_interview_qs(type=INTERVIEW_TYPES.final),
            has_unfinished_final_interview=application_has_interview_qs(
                type=INTERVIEW_TYPES.final,
                state__in=(
                    Interview.STATES.assigned,
                    Interview.STATES.estimated,
                ),
            ),
            candidate_has_unfinished_interview=_candidate_has_interview_qs(
                type__in=(
                    INTERVIEW_TYPES.regular,
                    INTERVIEW_TYPES.aa,
                ),
                state__in=(
                    Interview.STATES.assigned,
                    Interview.STATES.estimated,
                ),
            ),
        )
    )


APPL_STAGE_TO_Q = {
    APPLICATION_HIRING_STAGES.proposed: Q(
        proposal_status=APPLICATION_PROPOSAL_STATUSES.undefined,
        has_regular_interview=False,
        has_final_interview=False,
    ),
    APPLICATION_HIRING_STAGES.team_is_interested: Q(
        proposal_status=APPLICATION_PROPOSAL_STATUSES.accepted,
        has_regular_interview=False,
        has_final_interview=False,
    ),
    # Есть очные секции, но не все они завершены
    # ИЛИ есть незавершенные финалы
    APPLICATION_HIRING_STAGES.interview_assigned: (
        Q(
            has_regular_interview=True,
            candidate_has_unfinished_interview=True,
        )
        | Q(has_unfinished_final_interview=True)
    ),
    # Нет очных секций, но есть финалы, и они завершены
    # ИЛИ есть очки, они завершены, и нет незавершенных финалов
    APPLICATION_HIRING_STAGES.interview_finished: (
        Q(
            has_regular_interview=False,
            has_final_interview=True,
            has_unfinished_final_interview=False,
        )
        | Q(
            has_regular_interview=True,
            candidate_has_unfinished_interview=False,
            has_unfinished_final_interview=False,
        )
    ),
}


def filter_applications_by_stage(qs, stage):
    if stage in APPL_STAGE_TO_Q:
        qs = _annotate_applications_qs_with_interview_flags(qs)
        return qs.filter(APPL_STAGE_TO_Q[stage])
    return qs.none()


def get_candidate_visible_interviews(candidate):
    # Берем все секции активного рассмотрения кандидата
    interviews = list(
        candidate.interviews
        .exclude(type=INTERVIEW_TYPES.final)
        .filter(
            state=Interview.STATES.finished,
            consideration__state=Consideration.STATES.in_progress,
        )
        .values('id', 'state', 'type')
    )
    states = {i['state'] for i in interviews}

    # Рассмотрение завершено, если нет секций в статусе assigned или estimated
    if Interview.STATES.assigned in states or Interview.STATES.estimated in states:
        is_consideration_completed = False
    else:
        is_consideration_completed = True

    # Отдаем секции завершенного рассмотрения или скрининги
    ids = [
        i['id'] for i in interviews
        if (
            is_consideration_completed
            or i['type'] in (INTERVIEW_TYPES.hr_screening, INTERVIEW_TYPES.screening)
        )
    ]
    # Делаем отдельный запрос по ids. Если делать все одним запросом, получается медленно
    # TODO: Понять, почему медленно.
    return Interview.unsafe.filter(id__in=ids)


def annotate_with_last_comment(qs):
    # Формируем author в json вида, идентичного получаемому из UserSerializer,
    # чтобы не запрашивать и не сериализовывать его вновь в ApplicationSerializer.
    # Это необходимо, т.к. весь last_comment сериализуется в json на этапе БД из-за RowToDict
    author_qs = (
        User.objects
        .filter(id=OuterRef('author_id'))
        .values(
            'id',
            'username',
            'gender',
            'is_dismissed',
            login=F('username'),
            firstname=F('first_name'),
            lastname=F('last_name'),
            fullname=Concat(F('first_name'), Value(' '), F('last_name'))
        )[:1]
    )
    last_comment_qs = (
        Message.unsafe
        .alive()
        .filter(
            application=OuterRef('id'),
            type__in=COMMENT_MESSAGE_TYPES._db_values,
        )
        .order_by('-created')
        .values(
            'id',
            'created',
            'html',
            'modified',
            author_=RowToDict(author_qs),
        )
    )[:1]

    return qs.annotate(
        last_comment=RowToDict(last_comment_qs),
    )


def annotate_with_comments_count(qs):
    return qs.annotate(
        comments_count=get_count_subquery(
            queryset=(
                Message.unsafe
                .alive()
                .filter(type__in=COMMENT_MESSAGE_TYPES._db_values)
            ),
            reverse_related_name='application',
        )
    )


def annotate_with_interviews_count(qs):
    interviews_counts_qs = (
        Interview.unsafe
        .alive()
        .filter(application=OuterRef('id'))
        .values('type')
        .annotate(count=Count('id'))
    )

    return qs.annotate(
        interviews_counts=RowsToList(interviews_counts_qs),
    )


def annotate_application_qs_for_serialization(qs):
    qs = annotate_with_last_comment(qs)
    qs = annotate_with_comments_count(qs)
    qs = annotate_with_interviews_count(qs)
    return qs


def count_applications_per_vacancy(qs):
    return dict(
        qs
        .values('vacancy_id')
        .annotate(count=Count('id'))
        .values_list('vacancy_id', 'count')
    )


def count_applications_per_hiring_stage(qs):
    qs = _annotate_applications_qs_with_interview_flags(qs)
    return dict(
        qs
        .annotate(
            stage=Case(
                default=Value(''),
                output_field=CharField(),
                *[When(q, then=Value(stage)) for stage, q in APPL_STAGE_TO_Q.items()]
            )
        )
        .values('stage')
        .annotate(count=Count('id'))
        .values_list('stage', 'count')
    )
