from dateutil.relativedelta import relativedelta
from django.contrib.auth import get_user_model
from django.db.models import Exists, OuterRef
from django.template.loader import render_to_string
from django.utils import timezone

from intranet.femida.src.celery_app import app
from intranet.femida.src.applications.helpers import active_applications_query
from intranet.femida.src.applications.models import Application
from intranet.femida.src.communications.controllers import bulk_create_internal_messages
from intranet.femida.src.interviews.choices import (
    APPLICATION_CLOSE_REASONS,
    APPLICATION_RESOLUTIONS,
)
from intranet.femida.src.notifications.applications import ApplicationBulkClosedNotification
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.users.models import get_robot_femida
from intranet.femida.src.utils.lock import locked_task


User = get_user_model()


@app.autoretry_task(max_retries=3)
def notify_vacancy_applications_closed_task(application_ids,
                                            reason=APPLICATION_CLOSE_REASONS.vacancy_closed,
                                            initiator_id=None):
    """
    Создаёт комменты о закрытии прет-ов и отправляет отбивки.
    Вынесено в отдельный таск, потому что мы не готовы
    генерить нотифайки балком в рантайме
    """
    initiator = User.objects.get(id=initiator_id) if initiator_id else get_robot_femida()
    applications = (
        Application.unsafe
        .filter(id__in=application_ids)
        # Все select/prefetch нужны, в основном,
        # для формирования заголовков писем
        .select_related(
            'candidate',
            'vacancy__profession',
        )
        .prefetch_related(
            'candidate__target_cities',
            'candidate__skills',
            'candidate__candidate_professions__profession',
            'candidate__responsibles',
            'vacancy__cities',
            'vacancy__skills',
        )
    )

    reason_to_template = {
        APPLICATION_CLOSE_REASONS.vacancy_closed: 'vacancy-closed.txt',
        APPLICATION_CLOSE_REASONS.forgotten: 'forgotten.txt',
        APPLICATION_CLOSE_REASONS.offer_rejected: 'offer-rejected.txt',
    }
    template = reason_to_template.get(reason, 'default.txt')
    text = render_to_string('email/applications/bulk-close/' + template)
    messages = bulk_create_internal_messages(applications, text, initiator=initiator)
    for message in messages:
        notification = ApplicationBulkClosedNotification(
            instance=message.application,
            initiator=initiator,
            message=message,
        )
        notification.send()


@app.task
@locked_task
def close_forgotten_applications_task():
    """
    Закрывает активные прет-ва, на которых нет офферов,
    и которые не обновлялись больше месяца
    """
    from intranet.femida.src.applications.controllers import bulk_close_applications

    today = timezone.now().replace(hour=0, minute=0, second=0, microsecond=0)
    month_ago = today - relativedelta(months=1)

    # Note: нам интересны только активные офферы,
    # но в данном случае это не важно,
    # потому что при удалении/отказе/закрытии оффера,
    # прет-ва и так закрываются автоматом
    has_offer_sq = Exists(
        Offer.unsafe
        .filter(application=OuterRef('id'))
        .values('id')
    )

    applications = (
        Application.unsafe
        .annotate(has_offer=has_offer_sq)
        .filter(active_applications_query)
        .filter(
            has_offer=False,
            modified__lte=month_ago,
        )
    )
    bulk_close_applications(
        queryset=applications,
        resolution=APPLICATION_RESOLUTIONS.consideration_archived,
        reason=APPLICATION_CLOSE_REASONS.forgotten,
    )
