from typing import Optional, List, Iterable

from intranet.femida.src.applications.controllers import create_application_gracefully
from intranet.femida.src.applications.exceptions import (
    ApplicationAlreadyExists,
    InactiveVacancyError,
)
from intranet.femida.src.applications.helpers import annotate_application_qs_for_serialization
from intranet.femida.src.candidates.choices import (
    CONSIDERATION_STATUSES,
)
from intranet.femida.src.candidates.helpers import (
    add_profession_from_vacancy,
    add_skills_from_vacancy,
    blank_modify_candidate,
)
from intranet.femida.src.candidates.considerations.controllers import (
    get_or_create_relevant_consideration,
)
from intranet.femida.src.communications.choices import MESSAGE_TYPES
from intranet.femida.src.communications.controllers import (
    update_or_create_internal_message,
    bulk_create_internal_messages,
)
from intranet.femida.src.notifications.candidates import CandidateApplicationBulkCreatedNotification
from intranet.femida.src.oebs.choices import BUDGET_POSITION_STATUSES

from intranet.femida.src.interviews.choices import (
    APPLICATION_STATUSES,
    APPLICATION_SOURCES,
    APPLICATION_RESOLUTIONS,
    APPLICATION_PROPOSAL_STATUSES,
)
from intranet.femida.src.interviews.models import Application
from intranet.femida.src.oebs.api import get_budget_position, BudgetPositionError
from intranet.femida.src.vacancies.choices import VACANCY_STATUSES, ACTIVE_VACANCY_STATUSES
from intranet.femida.src.core.workflow import Workflow, Action, WorkflowError
from intranet.femida.src.notifications.applications import (
    ApplicationActivatedNotification,
    ApplicationClosedNotification,
    AcceptProposalNotification,
    RejectProposalNotification,
)
from intranet.femida.src.utils.switches import is_candidate_main_recruiter_enabled


class CreateMessageMixin:
    """
    Миксин для экшенов, где нужно создавать сообщение в коммуникации
    """
    def create_message(self, comment):
        message = None
        if comment:
            message_data = {
                'application_id': self.instance.id,
                'candidate_id': self.instance.candidate_id,
                'type': MESSAGE_TYPES.internal,
                'text': comment,
            }
            message = update_or_create_internal_message(
                data=message_data,
                initiator=self.user,
                need_notify=False,
                need_modify_candidate=False,
            )
        return message


class ApplicationBaseAction(Action):

    def get_annotated_instance(self):
        qs = (
            Application.unsafe
            .filter(id=self.instance.id)
        )
        return annotate_application_qs_for_serialization(qs).first()


class BulkCreateAction(ApplicationBaseAction):

    def __init__(self, wf, **kwargs):
        super().__init__(wf, **kwargs)
        self.candidate = wf.candidate
        self.vacancy = wf.vacancy
        self.fail_silently = False

    def has_permission(self):
        """
        Этот экшен контексто-зависим.
        Если мы не знаем ни кандидата, ни вакансии, то теоретически может создать любой.
        Если знаем вакансию, то можем ограничить по нанимающей команде.
        """
        if self.vacancy is None:
            return True
        return self._has_vacancy_permissions(self.vacancy)

    def perform(self, **params) -> Iterable[Application]:
        vacancies = params.pop('vacancies', [])
        self.fail_silently = len(vacancies) > 1
        interviews = params.pop('interviews', None)

        candidate_data = {
            'source': params.pop('source', None),
            'source_description': params.pop('source_description', None),
        }
        responsibles = self._get_candidate_new_responsibles(vacancies)
        if is_candidate_main_recruiter_enabled():
            candidate_data['main_recruiter'] = responsibles[0]
            candidate_data['responsibles'] = responsibles[1:]
        else:
            candidate_data['responsibles'] = responsibles

        params['consideration'] = self._get_or_create_relevant_consideration(candidate_data)

        applications = self._create_bulk(vacancies, **params)
        applications = self._get_application_qs(applications)
        messages = self._create_messages(applications, params['comment'])

        blank_modify_candidate(params.get('candidate'))

        notification = CandidateApplicationBulkCreatedNotification(
            instance=params['candidate'],
            initiator=self.user,
            applications=applications,
            message=messages[0] if messages else None,
            interviews=interviews,
        )
        notification.send()

        return applications

    def _create_bulk(self, vacancies, **params) -> List[Application]:
        """
        Для каждой вакансии из `vacancies` создаёт претендента на основе данных из `params`
        """
        applications = []
        for vacancy in vacancies:
            instance = self._create_single(vacancy=vacancy, **params)
            if instance:
                applications.append(instance)
        return applications

    def _create_single(self, **params) -> Optional[Application]:
        """
        Создаёт претендента на `vacancy` на основе `params`
        """
        vacancy = params['vacancy']
        if not self._has_vacancy_permissions(vacancy):
            self._raise_vacancy_error('vacancy_permission_denied', vacancy)
            return

        if not self.user.is_recruiter and self.user in vacancy.team:
            params['status'] = APPLICATION_STATUSES.in_progress
            params['source'] = APPLICATION_SOURCES.request
            params['proposal_status'] = APPLICATION_PROPOSAL_STATUSES.accepted
        if params.pop('create_activated', False):
            params['status'] = APPLICATION_STATUSES.in_progress

        try:
            instance = create_application_gracefully(params, self.user)
        except InactiveVacancyError:
            self._raise_vacancy_error('vacancy_is_not_in_progress', vacancy, force=True)
        except ApplicationAlreadyExists:
            self._raise_vacancy_error('application_already_exists', vacancy)
        else:
            add_profession_from_vacancy(instance.candidate, instance.vacancy)
            add_skills_from_vacancy(instance.candidate, instance.vacancy)
            return instance

    def _has_vacancy_permissions(self, vacancy):
        return (
            self.user.is_recruiter
            or self.user in vacancy.team
        )

    def _raise_vacancy_error(self, code, vacancy, *, force=False):
        self.raise_error(
            code=code,
            extra_context={
                'id': vacancy.id,
                'name': vacancy.name,
            },
            fail_silently=False if force else self.fail_silently,
        )

    def _get_candidate_new_responsibles(self, vacancies):
        if self.user.is_recruiter:
            return [self.user]
        else:
            return [v.main_recruiter for v in vacancies if v.main_recruiter]

    def _get_or_create_relevant_consideration(self, candidate_data):
        return get_or_create_relevant_consideration(
            candidate=self.candidate,
            initiator=self.user,
            candidate_data=candidate_data,
        )

    def _create_messages(self, applications, text):
        if not text:
            return []
        return bulk_create_internal_messages(
            applications=applications,
            initiator=self.user,
            text=text,
        )

    def _get_application_qs(self, applications: Iterable[Application]):
        appl_ids = [a.id for a in applications]
        applications = Application.unsafe.filter(id__in=appl_ids)
        applications = annotate_application_qs_for_serialization(applications)
        applications = (
            applications
            .select_related(
                'vacancy__profession',
            )
            .prefetch_related(
                'vacancy__cities',
                'vacancy__skills',
            )
        )
        return applications


class ActivateReopenActionBase(CreateMessageMixin, ApplicationBaseAction):

    def perform(self, **params):
        self.instance.status = APPLICATION_STATUSES.in_progress
        # Затираем неактуальную резолюцию при reopen
        self.instance.resolution = ''
        self.instance.save()
        blank_modify_candidate(self.instance.candidate)
        message = self.create_message(params['comment'])

        notification = ApplicationActivatedNotification(
            instance=self.instance,
            initiator=self.user,
            message=message,
            interviews=params.pop('interviews', None),
        )
        notification.send()

        return self.get_annotated_instance()


class ActivateAction(ActivateReopenActionBase):

    valid_statuses = (
        APPLICATION_STATUSES.draft,
    )

    def is_status_correct(self):
        return (
            super().is_status_correct()
            and self.instance.vacancy.status != VACANCY_STATUSES.closed
        )

    def has_permission(self):
        return self.user.is_recruiter


class CloseAction(CreateMessageMixin, ApplicationBaseAction):

    valid_statuses = (
        APPLICATION_STATUSES.draft,
        APPLICATION_STATUSES.in_progress,
    )

    def has_permission(self):
        return (
            self.user.is_recruiter
            or self.user in self.instance.vacancy.team
        )

    def perform(self, **params):
        self.instance.status = APPLICATION_STATUSES.closed
        self.instance.resolution = params['resolution']
        self.instance.save()
        blank_modify_candidate(self.instance.candidate)
        message = self.create_message(params['comment'])

        notification = ApplicationClosedNotification(
            instance=self.instance,
            initiator=self.user,
            message=message,
        )
        notification.send()

        return self.get_annotated_instance()


class ReopenAction(ActivateReopenActionBase):

    valid_statuses = (
        APPLICATION_STATUSES.closed,
    )

    def is_status_correct(self):
        return (
            super().is_status_correct()
            and self.instance.consideration.status == CONSIDERATION_STATUSES.in_progress
            and self.instance.vacancy.status in ACTIVE_VACANCY_STATUSES
        )

    def has_permission(self):
        return (
            self.user.is_recruiter
            or self.user in self.instance.vacancy.team
        )


class ChangeResolutionAction(CreateMessageMixin, ApplicationBaseAction):

    valid_statuses = (
        APPLICATION_STATUSES.closed,
    )

    def has_permission(self):
        return self.user.is_recruiter

    def perform(self, **params):
        self.instance.resolution = params['resolution']
        self.instance.save()
        blank_modify_candidate(self.instance.candidate)
        self.create_message(params['comment'])
        return self.get_annotated_instance()


class CreateOfferAction(Action):
    """
    Создание оффера на основе претендентства
    """
    valid_statuses = (
        APPLICATION_STATUSES.in_progress,
    )

    def is_status_correct(self):
        return (
            super().is_status_correct()
            and self.instance.vacancy.status == VACANCY_STATUSES.in_progress
        )

    def has_permission(self):
        return (
            self.user in self.instance.vacancy.recruiters
            or self.user.is_recruiting_manager
        )

    def perform(self, **params):
        from intranet.femida.src.offers.controllers import create_offer_from_application

        try:
            bp_data = get_budget_position(self.instance.vacancy.budget_position_id)
            valid_bp_statuses = (
                BUDGET_POSITION_STATUSES.vacancy,
                BUDGET_POSITION_STATUSES.offer,
            )
            if bp_data['status'] not in valid_bp_statuses:
                raise BudgetPositionError(message='budget_position_invalid_status')
        except BudgetPositionError as ex:
            raise WorkflowError(ex.message)

        self.instance.resolution = APPLICATION_RESOLUTIONS.offer_agreement
        self.instance.save()

        offer = create_offer_from_application(self.instance, self.user)

        self.instance.vacancy.status = VACANCY_STATUSES.offer_processing
        self.instance.vacancy.save()

        return offer


class ProposalActionBase(CreateMessageMixin, ApplicationBaseAction):

    notification_class = None

    valid_statuses = (
        APPLICATION_STATUSES.in_progress,
    )

    valid_resolutions = (
        '',
        APPLICATION_RESOLUTIONS.test_task_sent,
        APPLICATION_RESOLUTIONS.test_task_done,
        APPLICATION_RESOLUTIONS.invited_to_preliminary_interview,
    )

    def is_status_correct(self):
        return (
            super().is_status_correct()
            and self.instance.resolution in self.valid_resolutions
            and self.instance.proposal_status == APPLICATION_PROPOSAL_STATUSES.undefined
        )

    def has_permission(self):
        return (
            self.user.is_superuser
            or self.user in self.instance.vacancy.team
        )

    def _update_instance(self):
        raise NotImplementedError

    def perform(self, **params):
        self._update_instance()

        comment = params['comment']
        message = self.create_message(comment)
        blank_modify_candidate(self.instance.candidate)

        notification = self.notification_class(
            instance=self.instance,
            initiator=self.user,
            message=message,
        )
        notification.send()

        return self.get_annotated_instance()


class AcceptProposalAction(ProposalActionBase):

    notification_class = AcceptProposalNotification

    def _update_instance(self):
        self.instance.proposal_status = APPLICATION_PROPOSAL_STATUSES.accepted
        self.instance.save()


class RejectProposalAction(ProposalActionBase):

    notification_class = RejectProposalNotification

    def _update_instance(self):
        self.instance.status = APPLICATION_STATUSES.closed
        self.instance.resolution = APPLICATION_RESOLUTIONS.team_was_not_interested
        self.instance.proposal_status = APPLICATION_PROPOSAL_STATUSES.rejected
        self.instance.save()


class ApplicationWorkflow(Workflow):

    def __init__(self, instance, user, **kwargs):
        super().__init__(instance, user, **kwargs)
        self.vacancy = kwargs.get('vacancy')
        self.candidate = kwargs.get('candidate')

    ACTION_MAP = {
        'bulk_create': BulkCreateAction,
        'activate': ActivateAction,
        'close': CloseAction,
        'reopen': ReopenAction,
        'change_resolution': ChangeResolutionAction,
        'create_offer': CreateOfferAction,
        'accept_proposal': AcceptProposalAction,
        'reject_proposal': RejectProposalAction,
    }
