import functools
from collections import OrderedDict

from django.apps import AppConfig
from django.core.validators import ProhibitNullCharactersValidator


def errors_as_dict(self):
    def error_as_dict(err):
        error = {
            'code': err.code,
            'message': err.message,  # Добавил message для обратной совместимости со старым фронтом
        }
        if err.params:
            error['params'] = err.params
        return error

    errors = {}
    for key, value in self._errors.items():
        if key:
            key = key[0] + ''.join('[%s]' % k for k in key[1:])
        else:
            key = ''
        errors[key] = [error_as_dict(e) for e in value]
    return {'errors': errors}


def _sform_init_patch(func):
    """
    Monkey-patch для добавления параметров partial и context в sform
    """
    @functools.wraps(func)
    def inner(self, *args, **kwargs):
        self.partial = kwargs.pop('partial', False)
        self.context = kwargs.pop('context', {})
        self.partial_fields = kwargs.pop('partial_fields', ())
        func(self, *args, **kwargs)

    return inner


def _sform_clean_fields_patch(func):
    """
    Monkey-patch для частичной валидации в sform
    """
    @functools.wraps(func)
    def inner(self):
        if self.partial:
            self.fields = OrderedDict([
                (k, v) for k, v in self.fields.items()
                if k in self.data
            ])
        return func(self)

    return inner


def patch_sform():
    import sform
    sform.BaseSForm.__init__ = _sform_init_patch(sform.BaseSForm.__init__)
    sform.BaseSForm._clean_fields = _sform_clean_fields_patch(sform.BaseSForm._clean_fields)
    sform.BaseSForm.errors = property(errors_as_dict)


def patch_django_model_multiple_choice_field():
    # Note: в django есть косяк, из-за которого в ModelMultipleChoiceField
    # можно в качестве значения прокинуть nul (0x00), он его пропустит до БД,
    # а в случае с postgresql возникнет ошибка
    # ValueError('A string literal cannot contain NUL (0x00) characters.')
    # https://nda.ya.ru/t/C1DDYkMe4XHqNo
    from django import forms

    def check_for_null(func):
        @functools.wraps(func)
        def inner(self, value):
            validator = ProhibitNullCharactersValidator()
            for v in value:
                validator(v)
            return func(self, value)
        return inner

    forms.ModelMultipleChoiceField._check_values = check_for_null(
        forms.ModelMultipleChoiceField._check_values,
    )


def add_femida_mongo_transport():
    from kombu.transport import TRANSPORT_ALIASES
    TRANSPORT_ALIASES['femida_mongo'] = 'intranet.femida.src.celery.transport:FemidaMongoTransport'


class FemidaAppConfig(AppConfig):
    name = 'intranet.femida.src'

    def ready(self):
        patch_sform()
        patch_django_model_multiple_choice_field()
        add_femida_mongo_transport()
