import mimetypes
import hashlib
import uuid

from functools import partial

from django.conf import settings
from django.db import models
from model_utils.models import TimeStampedModel

from intranet.femida.src.utils import tikaite


def get_unique_filename(filename, prefix):
    return '{prefix}/{name}_{hash}'.format(
        prefix=prefix,
        name=filename.lower(),
        hash=uuid.uuid4().hex,
    )


def upload_to(instance, filename, prefix):
    return get_unique_filename(filename, prefix=prefix)


def generate_sha1(file, blocksize=65536):
    hasher = hashlib.sha1()
    while True:
        buff = file.read(blocksize)
        if not buff:
            break
        hasher.update(buff)
    return hasher.hexdigest()


class Attachment(TimeStampedModel):

    uploader = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        related_name='attachments',
        on_delete=models.PROTECT,
        blank=True,
        null=True,
    )
    name = models.CharField(max_length=255, blank=True, default='')
    size = models.IntegerField(blank=True, null=True)
    content_type = models.CharField(max_length=255, default='', blank=True)
    attached_file = models.FileField(
        max_length=512,
        upload_to=partial(upload_to, prefix='attachments'),
    )
    sha1 = models.CharField(max_length=40, default='', blank=True, db_index=True)
    # Сырой текст, извлеченный с помощью tikaite
    text = models.TextField(blank=True, null=True)

    def extract_text(self):
        try:
            self.text = tikaite.extract_text(
                mds_path=self.attached_file.name,
                mimetype=self.content_type,
            )
            # Tikaite может обработать файл с бинарными вставками как текст (например, .eml),
            # и тогда он вернет эти бинарные вставки в изначальном виде.
            # В том числе недопустимые для Postgres NUL characters
            if '\x00' in self.text:
                self.text = None
        except (tikaite.NotParsedError, tikaite.UnsupportedMediaTypeError):
            # Если tikaite не смог распарсить, или формат не поддерживается, нет смысла
            # обрабатывать это или ретраить.
            self.text = None

    def save(self, *args, **kwargs):
        if not self.sha1:
            self.sha1 = generate_sha1(self.attached_file.file)
        if not self.size:
            self.size = self.attached_file.size
        if not self.content_type:
            self.content_type = mimetypes.guess_type(self.name)[0] or ''
        super().save(*args, **kwargs)

    def delete(self, using=None, keep_parents=False):
        result = super().delete(using, keep_parents)
        self.attached_file.delete()
        return result

    def __str__(self):
        return 'Attachment {}: {}'.format(self.id, self.name)
