import json

from datetime import datetime, timedelta

from constance import config
from django.db import connection

from intranet.femida.src.beamery import s3
from intranet.femida.src.staff.models import StaffSync


class BeameryUploader:
    """
    Загрузчик данных из Фемиды в Beamery.
    На данный момент поддержана только загрузка через s3,
    в дальнейшем перейдём на YMQ.
    """
    def __init__(self, model, dry_run=False):
        self.model = model
        self.dry_run = dry_run

    def upload(self, delta=True):
        func = self._upload_delta_to_s3 if delta else self._upload_all_to_s3
        return func()

    def _get_filename_prefix_for_s3(self):
        model = self.model.upper()
        dt = datetime.now().isoformat(timespec='seconds')
        return f'{model}-{dt}'

    def _upload_chunks_to_s3(self, sql, **sql_params):
        """
        Чанками получает данные через sql-функцию
        и как есть выгружает в Beamery через s3 (не боевое решение).
        """
        filename_prefix = self._get_filename_prefix_for_s3()
        chunk = 0
        chunk_size = config.BEAMERY_S3_CHUNK_SIZE
        max_chunks_count = config.BEAMERY_S3_MAX_CHUNKS_COUNT
        id__gt = 0

        with connection.cursor() as cursor:
            while not max_chunks_count or chunk < max_chunks_count:
                cursor.execute(sql, dict(
                    limit=chunk_size,
                    id__gt=id__gt,
                    **sql_params,
                ))
                result, id__gt = cursor.fetchone()
                if not result:
                    break

                data = json.dumps(result).encode('utf-8')
                filename = f'{filename_prefix}-{chunk}.json'
                if self.dry_run:
                    print(f'BeameryUploader: uploading {self.model} ({chunk_size}, {id__gt})')
                else:
                    s3.upload(data, filename)
                chunk += 1

    def _upload_all_to_s3(self):
        sql = f'SELECT * FROM get_{self.model}_for_beamery(%(limit)s, %(id__gt)s)'
        with StaffSync.sync(f'beamery_s3_{self.model}'):
            self._upload_chunks_to_s3(sql)

    def _upload_delta_to_s3(self):
        lag_minutes = 3  # небольшой запас времени, чтобы не терять данные из-за долгих транзакций
        sql = (
            f'SELECT * FROM get_delta_{self.model}_for_beamery'
            f'(%(limit)s, %(id__gt)s, %(modified__gte)s)'
        )
        with StaffSync.sync(f'beamery_s3_{self.model}') as synced_at:
            modified__gte = synced_at - timedelta(minutes=lag_minutes)
            self._upload_chunks_to_s3(sql, modified__gte=modified__gte)
