import logging
import re

from typing import Optional, Dict
from urllib.parse import urlparse, parse_qs
from requests.exceptions import ConnectionError, Timeout

from django.conf import settings
from ids.exceptions import BackendError

from intranet.femida.src.calendar import exceptions


logger = logging.getLogger(__name__)

CALENDAR_DT_FORMAT = '%Y-%m-%dT%H:%M:%SZ'


def get_event_id_by_url(url: str) -> Optional[int]:
    """
    Возвращает event_id по URL встречи Календаря

    URL бывает 2х видов:
    1. /event/?event_id=1
    2. /event/1
    """
    return (
        _get_event_id_by_new_url(url)
        or _get_event_id_by_old_url(url)
    )


def _get_event_id_by_new_url(url: str) -> Optional[int]:
    rgx = re.compile(r'/event/(?P<event_id>\d+)')
    match = rgx.search(url)
    result = match.groupdict() if match else {}
    if 'event_id' in result:
        return int(result['event_id'])
    return None


def _get_event_id_by_old_url(url: str) -> Optional[int]:
    event_url_query = urlparse(url).query
    params = parse_qs(event_url_query)
    event_id = params.get('event_id', []) or params.get('show_event_id', [])

    if len(event_id):
        event_id = event_id[0]
        try:
            return int(event_id)
        except ValueError:
            return None
    return None


def get_event_url(room: Dict) -> str:
    code = room.get('email', '').split('@')[0]
    if code:
        return f'{settings.STAFF_URL}map/#/conference_room/name_exchange/{code}'


def handle_calendar_ids_error(exc: BackendError, strict: bool, event_id: int):
    if not _is_connection_error(exc):
        raise
    logger.exception('Calendar API is not responding for event id `%s`', event_id)
    if strict:
        raise exceptions.CalendarError


def handle_calendar_bad_response(error: dict, strict: bool, event_id: int):
    error_code = error['name']
    logger.warning(
        'Calendar API responded with error for event id `%s`.'
        'Error code: %s, message: %s', event_id, error_code, error['message']
    )
    if strict:
        if error_code == 'event-not-found':
            raise exceptions.EventDoesNotExist
        elif error_code == 'no-permissions-for-event-action':
            raise exceptions.EventIsNotAvailable
        elif error_code == 'event-modified':
            raise exceptions.EventAlreadyModified
        raise exceptions.CalendarError


def _is_connection_error(exc: BackendError):
    connection_errors = (ConnectionError, Timeout)
    return exc.args and isinstance(exc.args[0], connection_errors)
