from datetime import date, datetime

from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.db.models.functions import Lower
from rest_framework import serializers

from intranet.femida.src.candidates.choices import (
    CANDIDATE_DEGREES,
    CONTACT_TYPES,
    LINKED_ENTITY_TYPES,
)
from intranet.femida.src.core.models import City
from intranet.femida.src.hh.models import HHEntityLink
from intranet.femida.src.offers.choices import SOURCES
from intranet.femida.src.skills.models import Skill


# TODO: Возможно есть еще какие-то типы
CONTACT_TYPE_MAP = {
    'phone': CONTACT_TYPES.phone,
    'email': CONTACT_TYPES.email,
    'skype': CONTACT_TYPES.skype,
    'telegram': CONTACT_TYPES.telegram,
}


class AHCandidateContactSerializer(serializers.Serializer):

    type = serializers.SerializerMethodField()
    account_id = serializers.CharField(source='value')

    def get_type(self, obj):
        return CONTACT_TYPE_MAP.get(obj['type'], CONTACT_TYPES.other)


class AHCandidateSiteSerializer(serializers.Serializer):

    type = serializers.ReadOnlyField(default=CONTACT_TYPES.other)
    account_id = serializers.CharField(source='value')


class AHCandidateEducationSerializer(serializers.Serializer):

    institution = serializers.CharField(source='name')
    faculty = serializers.CharField()
    end_date = serializers.SerializerMethodField()
    degree = serializers.CharField(default=CANDIDATE_DEGREES.unknown)

    def get_end_date(self, obj):
        try:
            return date(int(obj['end']), 1, 1)
        except (ValueError, TypeError):
            return None


class AHCandidateJobSerializer(serializers.Serializer):

    id = serializers.CharField(default=None)
    employer = serializers.SerializerMethodField()
    position = serializers.CharField()
    start_date = serializers.SerializerMethodField()
    end_date = serializers.SerializerMethodField()
    comment = serializers.CharField(source='description')

    def get_employer(self, obj):
        # Иногда company приходит как пустой dict
        return obj.get('company', {}).get('name', '')

    def _strptime(self, value):
        formats = [
            '%Y',
            '%Y-%m',
            '%Y-%m-%d',
        ]
        for f in formats:
            try:
                return datetime.strptime(value, f).date()
            except ValueError:
                pass

    def get_start_date(self, obj):
        if obj.get('start'):
            return self._strptime(obj['start'])

    def get_end_date(self, obj):
        if obj.get('end'):
            return self._strptime(obj['end'])


class AmazingHiringCandidateSerializer(serializers.Serializer):

    last_name = serializers.SerializerMethodField()
    first_name = serializers.SerializerMethodField()
    birthday = serializers.DateField()
    city = serializers.SerializerMethodField()
    target_cities = serializers.SerializerMethodField()
    contacts = serializers.SerializerMethodField()
    educations = AHCandidateEducationSerializer(many=True)
    jobs = AHCandidateJobSerializer(source='positions', many=True)
    skills = serializers.SerializerMethodField()
    source = serializers.CharField(default=SOURCES.external_website)
    source_description = serializers.SerializerMethodField(method_name='_get_ah_url')

    def get_last_name(self, obj):
        # Бывает в name только одно слово
        name = obj['name'].split()
        return name[1] if len(name) > 1 else None

    def get_first_name(self, obj):
        # Бывает в name ни одного слова
        name = obj['name'].split()
        return name[0] if len(name) > 0 else None

    def get_city(self, obj):
        locations = obj.get('locations', [])
        if locations:
            return locations[0]['name']

    def get_target_cities(self, obj):
        keys = {l['id'] for l in obj['locations']}
        content_type = ContentType.objects.get_for_model(City)
        return set(
            HHEntityLink.objects
            .filter(
                type=LINKED_ENTITY_TYPES.amazing_hiring,
                content_type=content_type,
                key__in=keys,
            )
            .values_list('object_id', flat=True)
        )

    def get_contacts(self, obj):
        contacts = AHCandidateContactSerializer(obj['contacts'], many=True).data
        contacts.extend(AHCandidateSiteSerializer(obj['links'], many=True).data)
        contacts.append({
            'type': CONTACT_TYPES.ah,
            'account_id': self._get_ah_url(obj),
        })
        main_contact_types = ('email', 'phone', 'skype')
        found_main_contact_types = set()
        for contact in contacts:
            contact_type = contact['type']
            if contact_type in main_contact_types and contact_type not in found_main_contact_types:
                found_main_contact_types.add(contact_type)
                contact['is_main'] = True
            else:
                contact['is_main'] = False

        return contacts

    def get_skills(self, obj):
        keys = {s['name'].lower() for s in obj['skills']}
        content_type = ContentType.objects.get_for_model(Skill)
        object_id_by_key = dict(
            HHEntityLink.objects
            .filter(
                type=LINKED_ENTITY_TYPES.amazing_hiring,
                content_type=content_type,
                key__in=keys,
            )
            .values_list('key', 'object_id')
        )
        skill_ids = set(object_id_by_key.values())

        other_keys = keys - object_id_by_key.keys()
        other_skill_ids = (
            Skill.objects
            .annotate(name_lower=Lower('name'))
            .filter(name_lower__in=other_keys)
            .values_list('id', flat=True)
        )
        skill_ids.update(other_skill_ids)
        return skill_ids

    def _get_ah_url(self, obj):
        return 'https://{host}/profiles/{id}'.format(
            host=settings.AMAZING_HIRING_HOST,
            id=obj['id'],
        )
