from functools import wraps
from typing import Iterable

from intranet.femida.src.actionlog.models import actionlog, SNAPSHOT_REASONS
from intranet.femida.src.candidates.choices import CANDIDATE_STATUSES
from intranet.femida.src.candidates.contacts import normalize_contact
from intranet.femida.src.candidates.controllers import (
    update_candidate_tags,
    update_or_create_candidate,
)
from intranet.femida.src.candidates.models import (
    CandidateCity,
    CandidateContact,
    CandidateProfession,
    CandidateSkill,
    CandidateEducation,
    CandidateJob,
)
from intranet.femida.src.candidates.tasks import (
    run_duplicates_search_for_candidate,
    create_candidate_attachment_by_url_task,
)
from intranet.femida.src.communications.controllers import update_or_create_note
from intranet.femida.src.core.controllers import update_instance
from intranet.femida.src.core.models import Tag
from intranet.femida.src.core.signals import post_bulk_create


def bulk_created(func):
    """
    Декоратор для функций с балковыми записями в БД.
    После выполнения функции func отправляет сигнал post_bulk_create
    для записи событий в actionlog.
    Ожидается, что func возвращает список созданных объектов.
    """
    @wraps(func)
    def inner(*args, **kwargs):
        created = func(*args, **kwargs)
        if created:
            post_bulk_create.send(
                sender=created[0].__class__,
                queryset=created,
            )
        return created
    return inner


def _get_missing_target_cities(candidate, new_target_cities):
    return set(new_target_cities) - set(candidate.target_cities.all())


def _get_missing_contacts(candidate, new_contacts, *, respect_inactive=False) -> Iterable[dict]:
    existing_contacts = candidate.contacts.all()
    if not respect_inactive:
        existing_contacts = existing_contacts.filter(is_active=True)

    existing_main_contact_types = set()
    existing_contacts_set = set()

    for contact in existing_contacts:
        existing_contacts_set.add((contact.type, contact.account_id))
        if contact.normalized_account_id:
            existing_contacts_set.add((contact.type, contact.normalized_account_id))
        if contact.is_main:
            existing_main_contact_types.add(contact.type)

    missing_contacts = []

    for contact in new_contacts:
        normalized = normalize_contact(contact['type'], contact['account_id'])
        contact['normalized_account_id'] = normalized or ''

        is_contact_already_exists = (
            (normalized and (contact['type'], normalized) in existing_contacts_set)
            or (contact['type'], contact['account_id']) in existing_contacts_set
        )
        if is_contact_already_exists:
            continue
        if contact['type'] in existing_main_contact_types:
            contact['is_main'] = False
        missing_contacts.append(contact)

    return missing_contacts


def _get_missing_professions(candidate, new_professions):
    existing_profession_ids = set(
        candidate.candidate_professions
        .values_list('profession_id', flat=True)
    )
    return [
        p for p in new_professions
        if p['profession'].id not in existing_profession_ids
    ]


def _get_missing_skills(candidate, new_skills):
    existing_skill_ids = set(
        candidate.candidate_skills
        .values_list('skill_id', flat=True)
    )
    return [s for s in new_skills if s.id not in existing_skill_ids]


def _get_missing_tags(candidate, new_tags, *, respect_inactive=False) -> Iterable[Tag]:
    existing_tag_ids = candidate.candidate_tags.values_list('tag', flat=True)
    if not respect_inactive:
        existing_tag_ids = existing_tag_ids.filter(is_active=True)
    existing_tag_ids = set(existing_tag_ids)
    return [t for t in new_tags if t.id not in existing_tag_ids]


def _create_tags(tags):
    tags = set(tags)
    Tag.objects.bulk_create(
        objs=[Tag(name=name) for name in tags],
        ignore_conflicts=True,
    )
    return Tag.objects.filter(name__in=tags)


@bulk_created
def _create_target_cities(candidate, data):
    return CandidateCity.objects.bulk_create([
        CandidateCity(
            candidate=candidate,
            city=city,
        )
        for city in data
    ])


@bulk_created
def _create_contacts(candidate, data):
    return CandidateContact.objects.bulk_create([
        CandidateContact(candidate=candidate, **kw)
        for kw in data
    ])


@bulk_created
def _create_professions(candidate, data):
    return CandidateProfession.objects.bulk_create([
        CandidateProfession(candidate=candidate, **kw)
        for kw in data
    ])


@bulk_created
def _create_skills(candidate, data):
    return CandidateSkill.objects.bulk_create([
        CandidateSkill(
            candidate=candidate,
            skill=skill,
            confirmed_by=[],
        )
        for skill in data
    ])


@bulk_created
def _create_educations(candidate, data):
    return CandidateEducation.objects.bulk_create([
        CandidateEducation(candidate=candidate, **kw)
        for kw in data
    ])


@bulk_created
def _create_jobs(candidate, data):
    return CandidateJob.objects.bulk_create([
        CandidateJob(candidate=candidate, **kw)
        for kw in data
    ])


def _create_note(candidate, text, author):
    if not text:
        return
    data = {
        'candidate': candidate,
        'text': text,
    }
    return update_or_create_note(
        data=data,
        initiator=author,
        need_notify=False,
        need_modify_candidate=False,
    )


def _create_note_from_answers(candidate, answers, author):
    if not answers:
        return
    template = '**{question}**\n%%{answer}%%'
    text = '\n\n'.join(template.format(**i) for i in answers)
    return _create_note(candidate, text, author)


def _create_candidate_attachments_by_urls(candidate, urls):
    for url in urls:
        create_candidate_attachment_by_url_task.delay(candidate.id, url)


def create_candidate(data):
    note_author = data.pop('note_author', None)
    note = data.pop('note', None)
    answers = data.pop('answers', [])
    attachment_urls = data.pop('attachment_link', [])
    data['tags'] = _create_tags(data.get('tags', []))
    candidate = update_or_create_candidate(data)
    candidate.status = CANDIDATE_STATUSES.closed
    candidate.save()
    _create_note(candidate, note, note_author)
    _create_note_from_answers(candidate, answers, note_author)
    _create_candidate_attachments_by_urls(candidate, attachment_urls)
    return candidate


# Внимание! Костыль.
# TODO: Поресерчить, обобщить и убрать.
def _not_empty(k, v):
    """
    Для поля vacancies_mailing_agreement отдельный кейс, так как
    оно имеет тип Optional[bool], который населен тремя значениями.
    Они все валидные и должны быть правильно обработаны.
    Для остальных полей применяем старую логику.
    """
    if k == 'vacancies_mailing_agreement':
        return v is not None
    else:
        return v


def merge_into_candidate(candidate, data, *, respect_inactive=False):
    """
    :param candidate: кандидат, в которого нужно залить новые данные
    :param data: сами новые данные
    :param respect_inactive: не восстанавливать ранее помеченные неактивными контакты, теги и т.д.
      – т.е. не создавать точно такие же активные, если есть "удаленные"
    """
    keep_modification_time = data.pop('keep_modification_time', False)

    # Очищаем все пустые значения
    data = {k: v for k, v in data.items() if _not_empty(k, v)}

    # Получаем все новые связанные данные
    target_cities = data.pop('target_cities', [])
    contacts = data.pop('contacts', [])
    candidate_professions = data.pop('candidate_professions', [])
    skills = data.pop('skills', [])
    attachment_urls = data.pop('attachment_link', [])
    tags = data.pop('tags', [])
    educations = data.pop('educations', [])
    jobs = data.pop('jobs', [])

    # Данные для заметки
    note_author = data.pop('note_author', None)
    note = data.pop('note', None)
    answers = data.pop('answers', [])

    update_instance(candidate, data, keep_modification_time=keep_modification_time)
    tags = _create_tags(tags)

    # Получаем для всех связанных данных только те, которых ещё нет в БД
    target_cities = _get_missing_target_cities(candidate, target_cities)
    contacts = _get_missing_contacts(candidate, contacts, respect_inactive=respect_inactive)
    candidate_professions = _get_missing_professions(candidate, candidate_professions)
    skills = _get_missing_skills(candidate, skills)
    tags = _get_missing_tags(candidate, tags, respect_inactive=respect_inactive)

    # Создаем недостающие записи
    _create_target_cities(candidate, target_cities)
    _create_contacts(candidate, contacts)
    _create_professions(candidate, candidate_professions)
    _create_skills(candidate, skills)
    update_candidate_tags(candidate, tags, delete_missing=False)

    # Добавляем новые данные
    _create_educations(candidate, educations)
    _create_jobs(candidate, jobs)
    _create_note(candidate, note, note_author)
    _create_note_from_answers(candidate, answers, note_author)
    _create_candidate_attachments_by_urls(candidate, attachment_urls)

    # Сразу запускаем поиск других дублей
    run_duplicates_search_for_candidate.delay(candidate.id)

    return candidate


def beamery_create_candidate(data):
    assert actionlog.is_initialized(), 'function must be called within an actionlog'
    # Пока не умеем обрабатывать статус, ничего с ним не делаем
    data.pop('status', None)
    return create_candidate(data)


def beamery_merge_into_candidate(candidate, data):
    assert actionlog.is_initialized(), 'function must be called within an actionlog'
    status = data.pop('status', None)
    candidate = merge_into_candidate(candidate, data, respect_inactive=True)

    # Если статус не соответствует нашему, добавляем кандидата в actionlog,
    # чтобы для него сработал touch_candidates_task после коммита транзакции
    if status and status != candidate.status:
        actionlog.add_object(candidate, SNAPSHOT_REASONS.nothing)

    return candidate
